package org.lsst.ccs.utilities.ccd;

import java.awt.Dimension;

/**
 * An class to represent a CCD Segment.
 * Each Segment is defined by its geometry constants, the parameters for reading
 * it out (prescan and overscans) and the order in which it is read out. 
 * 
 * A Segment also has a channel number which is used to write the extended
 * headers in fits files.
 * 
 * These quantities are as specified in LCA-13501
 *
 * @author The LSST CCS Team
 */
public class Segment extends Geometry {

    public static enum SegmentsRotation {
        ZERO, NIGHTY, ONE_EIGHTY, TWO_SEVENTY;
    }

    private final int channel;

    /**
     * Create a Segment instance.
     * TO-DO Where is channel defined? Ask Seth where we can find the definition.
     * 
     * @param segmentGeometryConstants The Segment Geometry information
     * @param readOutOrder the read out order for this segment
     * @param channel The Channel number of this Segment.
     */
    Segment(SegmentGeometryConstants segmentGeometryConstants, int channel) {
        super("Seg", new Dimension(segmentGeometryConstants.getSegmentParallelActiveSize(),segmentGeometryConstants.getSegmentSerialActiveSize()));

        this.channel = channel;
    }
    
    /**
     * The Segment channel number.
     * 
     * @return The Segment channel
     */
    public int getChannel() {
        return channel;
    }

    /**
     * The Segment active serial size in pixels.
     * 
     * @return The Segment active serial size.
     */
    public int getSegmentSerialActiveSize() {
        return getHeight();
    }

    /**
     * The Segment active parallel size in pixels.
     * 
     * @return The Segment parallel serial size.
     */
    public int getSegmentParallelActiveSize() {
        return getWidth();
    }
    
    //Segments don't allow children
    @Override
    protected void addGeometryToGrid(Geometry child, int p, int s) {
        throw new UnsupportedOperationException("Cannot add a geometry to a Segment.");
    }

    /**
     * Utility function to create CCDSegments by type for a given CCDType
     * and a SegmenReadOutOrder. Segments also have a channel id that is used
     * when creating fits files.
     * 
     * @param ccdType      The CCD type
     * @param readoutOrder The SegmentReadOutOrder
     * @param channel      The channel id
     *
     * @return The Segment for the provided type.
     */
    static Segment createCCDSegment(CCDType ccdType,int channel, SegmentsRotation rotation) {
        SegmentGeometryConstants geomConstants = ccdType.getCCDGeometryConstants().getSegmentGeometryConstraint();
        if ( rotation == SegmentsRotation.NIGHTY || rotation == SegmentsRotation.TWO_SEVENTY ) {
            geomConstants = new SegmentGeometryConstants(geomConstants.getSegmentSerialActiveSize(), geomConstants.getSegmentParallelActiveSize(), geomConstants.getSegmentSerialPrescanSize());
        }
        
        return new Segment(geomConstants, channel);
    }
    static Segment createCCDSegment(CCDType ccdType, int channel) {
        return createCCDSegment(ccdType, channel, SegmentsRotation.ZERO);
    }
}
