package org.lsst.ccs.utilities.image;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;
import org.lsst.ccs.utilities.logging.Logger;

/**
 * Utility class for the generation of header specifications objects.
 * These objects create a map between fits header keywords and quantities that
 * are loaded at runtime.
 * Each spec file is linked to a named header. Default name headers that must be
 * provided when writing fits files are "primary" for the primary header and
 * "extended" for any data header file extension, i.e. extensions containing
 * image data.
 * Any additional header name (beside "primary" and "extended") will create an
 * additional extension in which the fits keywords are stored in a binary table.
 *
 * @author The LSST CCS Team
 */
public class FitsHeadersSpecificationsBuilder {

    
    private final Map<String, HeaderSpecification> headerSpecificationsMap = new HashMap<>();
    public static final Logger log = Logger.getLogger("org.lsst.ccs.utilities.image");

    
    /**
     * Add the a header specification file. This information will be used to 
     * generate the Header configuration map. The name of the file (excluding 
     * the .spec extension if provided) will be used as
     * the name of the header extension. 
     * If a Header already exists for the provided name, the new header lines
     * will be superimposed with the existing ones.
     * 
     * @param fileName The name of the file to be loaded. The extension may be omitted.
     *                 No path information is to be provided. The file will be loaded
     *                 as a Bootstrap resource.
     */
    public final void addSpecFile(String fileName) {
        readHeaderSpecification(fileName, null);
    }

    /**
     * Add the a header specification file. This information will be used to 
     * generate the Header configuration map. The name of the file (excluding 
     * the .spec extension if provided) will be used as
     * the name of the header extension unless a corresponding header name is provided.
     * If a Header already exists for the provided name, the new header lines
     * will be superimposed with the existing ones.
     * @param fileName The name of the file to be loaded. The extension may be omitted.
     *                 No path information is to be provided. The file will be loaded
     *                 as a Bootstrap resource.
     * @param headerName The headers name for which this spec file is added
     *                   If no header name is provided, the file name will be used.
     * 
     */
    public final void addSpecFile(String fileName, String headerName) {
        readHeaderSpecification(fileName, headerName);
    }
        
    /**
     * Get the HeaderSpecifications as built from the read spec files.
     * @return The Map of HeaderSpecifications;
     */    
    public final Map<String, HeaderSpecification> getHeaderSpecifications() {
        return Collections.unmodifiableMap(headerSpecificationsMap);
    }

    private void readHeaderSpecification(String name, String headerName) throws RuntimeException {
        final String fileName = name.endsWith(".spec") ? name : name + ".spec";
        if ( headerName == null ) {
            headerName = fileName.replace(".spec", "");
        }
        try (InputStream bootstrapResource = BootstrapResourceUtils
                .getBootstrapResource(fileName)) {
            if (bootstrapResource == null) {
                log.severe("Failed to open spec file "+fileName);
                return;
            }
            HeaderSpecification desc = null;
            
            if ( headerSpecificationsMap.containsKey(headerName) ) {
                desc = headerSpecificationsMap.get(headerName);
                desc.loadHeaderSpecification(bootstrapResource);
            } else {            
                desc = new HeaderSpecification(headerName,
                    bootstrapResource);
                headerSpecificationsMap.put(headerName, desc);
            }
        } catch (IOException x) {
            throw new RuntimeException("Error reading " + fileName, x);
        }
    }
    
}
