
package org.lsst.ccs.utilities.image.patterns;

import java.awt.image.BufferedImage;
import java.awt.image.WritableRaster;
import java.util.Map;

/**
 * A PatternGenerator that creates checkered pattern with sinusoidal fluctuations
 * of a given amplitude over a baseline.
 * 
 * The allowed parameters for this patternGenerator are:
 *   - checkered_maxAmplitude: the maximum amplitude of the pattern. An integer from 0 to 256
 *   - checkered_baseline: the baseline of the pattern. An integer form 0 to 256
 *   - checkered_frequency: the frequency of the sinusoidal fluctuations
 * 
 * The idea for this code was taken from http://www.lac.inpe.br/JIPCookbook/1200-create-gl.jsp
 * 
 * @author turri
 */
public class CheckeredPatternGenerator extends PatternGenerator {
    
    CheckeredPatternGenerator() {
        super("checkered");
    }

    @Override
    public GeneratedImage generateImage(int width, int height, Map<String, String> parameters) {

        int checkered_baseline = getIntParameter("checkered_baseline", parameters, 128);
        int checkered_maxAmplitude = getIntParameter("checkered_maxAmplitude", parameters, 256);
        double checkered_frequency = getDoubleParameter("checkered_frequency", parameters, 32);

        int maxFluctuation = Math.max(checkered_maxAmplitude - checkered_baseline,0);
        
        int[][] data = new int[width][height];
        for (int h = 0; h < height; h++) {
            for (int w = 0; w < width; w++) {
                int value = checkered_baseline + (int) (maxFluctuation * Math.sin(w / checkered_frequency) * Math.sin(h / checkered_frequency)); // Weird sin pattern.
                data[w][h] = value;
            }
        }
        return new GeneratedImage(data,checkered_maxAmplitude);
    }
    
}
