package org.lsst.ccs.utilities.image;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.ShortBuffer;
import java.nio.charset.StandardCharsets;
import java.util.logging.Level;
import java.util.logging.Logger;
import nom.tam.fits.BasicHDU;
import nom.tam.fits.FitsException;
import nom.tam.fits.Header;
import nom.tam.fits.HeaderCard;
import static nom.tam.fits.header.Checksum.CHECKSUM;
import static nom.tam.fits.header.Checksum.DATASUM;
import nom.tam.util.AsciiFuncs;
import nom.tam.util.BufferedDataOutputStream;
import nom.tam.util.FitsIO;

/**
 * A replacement for the FitsCheckSum implemented in
 * {@link nom.tam.fits.utilities.FitsCheckSum}. This class is a
 * little more flexible, specifically it allows the checksum to be accumulated 
 * across several invocations of the {@link #updateChecksum} method, and allows
 * for the DATASUM keyword to be updated after data has been changed. It also 
 * provides a method for decoding an encoded checksum.
 * 
 * @author tonyj
 * @see <a href="http://arxiv.org/abs/1201.1345" target="@top">FITS Checksum Proposal</a>
 */
public class FitsCheckSum {

    private static final int CHECKSUM_BLOCK_SIZE = 4;
    private static final int CHECKSUM_STRING_SIZE = 16;
    private static final int ASCII_ZERO = 0x30;
    private static final Logger log = Logger.getLogger(FitsCheckSum.class.getName());

    private FitsCheckSum() {
    }
    /**
     * Encapsulation of a FITS check sum. 
     */
    public static class Checksum {

        private long hi;
        private long lo;

        public Checksum() {
            hi = 0;
            lo = 0;
        }

        private long getHi() {
            return hi;
        }

        private long getLo() {
            return lo;
        }

        private void update(long hi, long lo) {
            long hicarry = hi >> 16;
            long locarry = lo >> 16;
            while (hicarry > 0 || locarry > 0) {
                hi = (hi & 0xFFFF) + locarry;
                lo = (lo & 0xFFFF) + hicarry;
                hicarry = hi >> 16;
                locarry = lo >> 16;
            }
            this.hi = hi;
            this.lo = lo;
        }

        public long getCheckSum() {
            return (hi << 16) | lo;
        }

    }
    /**
     * Calculate the Seaman-Pence 32-bit 1's complement checksum over the byte
     * stream. The option to start from an intermediate checksum accumulated
     * over another previous byte stream is not implemented. The implementation
     * accumulates in two 64-bit integer values the two low-order and the two
     * high-order bytes of adjacent 4-byte groups. A carry-over of bits is never
     * done within the main loop (only once at the end at reduction to a 32-bit
     * positive integer) since an overflow of a 64-bit value (signed, with
     * maximum at 2^63-1) by summation of 16-bit values could only occur after
     * adding approximately 140G short values (=2^47) (280GBytes) or more. We
     * assume for now that this routine here is never called to swallow FITS
     * files of that size or larger. by R J Mathar
     * {@link nom.tam.fits.header.Checksum#CHECKSUM}
     * 
     * @param data
     *            the byte sequence
     * @return the 32bit checksum in the range from 0 to 2^32-1
     * @since 2005-10-05
     */
    public static long checksum(byte[] data) {
        return computeChecksum(ByteBuffer.wrap(data));
    }

    /**
     * Compute a FITS check sum from a ByteBuffer
     * @param data The ByteBuffer to use as a source of data
     * @return The computed check sum
     */
    public static long computeChecksum(ByteBuffer data) {
        return updateChecksum(data, new Checksum());
    }

    //AN ATTEMPT TO IMPLEMENT UPDATE CHECKSUM WITHOUG USING "asShortBuffer"
    //to simplify the ImageDataStream interface
//    public static long updateChecksumNew(ByteBuffer data, Checksum inOut) {
//        if (!(data.remaining() % 4 == 0)) {
//            throw new IllegalArgumentException("fits blocks always must be dividable by 4");
//        }
//        data.order(ByteOrder.BIG_ENDIAN);
//        long hi = inOut.getHi();
//        long lo = inOut.getLo();
//        while (data.hasRemaining()) {
//            short s1 = (short)((( data.get() & 0xFF) << 8) | (data.get() & 0xFF));
//            short s2 = (short)((( data.get() & 0xFF) << 8) | (data.get() & 0xFF));
//            hi += s1 & 0xffff;
//            lo += s2 & 0xffff;
//        }
//        inOut.update(hi, lo);
//        return inOut.getCheckSum();
//    }
//
    /**
     * Update a check sum from a ByteBuffer
     *
     * @param data The ByteBuffer to use as a source of data
     * @param inOut The previously accumulated check sum, updated in place
     * with the additional data.
     * @return The computed check sum
     */
    public static long updateChecksum(ByteBuffer data, Checksum inOut) {
        if (!(data.remaining() % 4 == 0)) {
            throw new IllegalArgumentException("fits blocks always must be dividable by 4");
        }
        data.order(ByteOrder.BIG_ENDIAN);
        ShortBuffer shortData = data.asShortBuffer();
        long hi = inOut.getHi();
        long lo = inOut.getLo();
        while (shortData.hasRemaining()) {
            hi += shortData.get() & 0xffff;
            lo += shortData.get() & 0xffff;
        }
        inOut.update(hi, lo);
        return inOut.getCheckSum();
    }
    /**
     * Encode a 32bit integer according to the Seaman-Pence proposal.
     * 
     * @see <a
     *      href="http://heasarc.gsfc.nasa.gov/docs/heasarc/ofwg/docs/general/checksum/node14.html#SECTION00035000000000000000">heasarc
     *      checksum doc</a>
     * @param c
     *            the checksum previously calculated
     * @param compl
     *            complement the value
     * @return the encoded string of 16 bytes.
     */
    public static String checksumEnc(final long c, final boolean compl) {
        return checksumEncode(compl ? ~c : c);
    }
    
    /**
     * Encode the given checksum, including the rotating the result right by one byte.
     * @param checksum
     * @return The encoded checksum, suitably encoded for use with the CHECKSUM header
     */
    public static String checksumEncode(final long checksum) {
        byte[] asc = new byte[CHECKSUM_STRING_SIZE];
        String exclude = ":;<=>?@[\\]^_`";

        final int offset = ASCII_ZERO;
        
        for (int i = 0; i < CHECKSUM_BLOCK_SIZE; i++) {
            // each byte becomes four
            final int byt = (int) (0xff & Long.rotateRight(checksum, (CHECKSUM_BLOCK_SIZE - i - 1) * 8));
            final int quotient = byt / CHECKSUM_BLOCK_SIZE + offset;
            final int remainder = byt % CHECKSUM_BLOCK_SIZE;
            int[] ch = new int[CHECKSUM_BLOCK_SIZE];
            for (int j = 0; j < CHECKSUM_BLOCK_SIZE; j++) {
                ch[j] = quotient;
            }

            ch[0] += remainder;

            for (int j = 0; j < CHECKSUM_BLOCK_SIZE; j += CHECKSUM_BLOCK_SIZE / 2) {
                while (exclude.indexOf(ch[j]) >= 0 || exclude.indexOf(ch[j + 1]) >= 0) {
                    ch[j]++;
                    ch[j + 1]--;
                }
            }

            for (int j = 0; j < CHECKSUM_BLOCK_SIZE; j++) {
                // assign the bytes
                asc[CHECKSUM_BLOCK_SIZE * j + i] = (byte) ch[j];
            }
        }
        // shift the bytes 1 to the right circularly.
        String resul = AsciiFuncs.asciiString(asc, CHECKSUM_STRING_SIZE - 1, 1);
        return resul.concat(AsciiFuncs.asciiString(asc, 0, CHECKSUM_STRING_SIZE - 1));
    }
    /**
     * Decodes an encoded checksum, the opposite of {@link #checksumEncode}
     * @param encoded The encoded checksum (16 character string)
     * @return The integer checksum.
     */
    public static long checksumDecode(String encoded) {
        byte[] chars = encoded.getBytes(StandardCharsets.US_ASCII);
        if (chars.length != CHECKSUM_STRING_SIZE) {
            throw new IllegalArgumentException("Encoded checksum must consist of 16 US-ASCII characters");
        }
        // Shift the bytes one to the left circularly
        byte tmp = chars[0];
        System.arraycopy(chars, 1, chars, 0, CHECKSUM_STRING_SIZE - 1);
        chars[CHECKSUM_STRING_SIZE - 1] = tmp;
        for (int i = 0; i < CHECKSUM_STRING_SIZE; i++) {
            chars[i] -= ASCII_ZERO;
        }
        ByteBuffer bb = ByteBuffer.wrap(chars);
        bb.order(ByteOrder.BIG_ENDIAN);
        long result = 0;
        result += bb.getInt();
        result += bb.getInt();
        result += bb.getInt();
        result += bb.getInt();
        return result & FitsIO.INTEGER_MASK;
    }
    /**
     * Apply an incremental update to the datasum, as a result of changing one of more records
     * @param header The header to update
     * @param dataSum The new checksum
     */    
    static void updateDataSum(Header header, long dataSum) {
        final HeaderCard dataSumCard = header.findCard(DATASUM);
        long oldDataSumCheckSum = FitsCheckSum.checksum(AsciiFuncs.getBytes(dataSumCard.toString()));
        final String oldValue = dataSumCard.getValue();
        final String newValue = Long.toString(dataSum);
        dataSumCard.setValue(newValue);
        long newDataSumCheckSum = FitsCheckSum.checksum(AsciiFuncs.getBytes(dataSumCard.toString()));
        // Updating the datasum also changes the CHECKSUM, so that must be recomputed too.
        // Checksum is affected both by the data changing, and by the DATASUM header changing
        // Fortunately this is (relatively) easy to do.
        long delta = (Long.parseLong(newValue) - Long.parseLong(oldValue))
                + (newDataSumCheckSum - oldDataSumCheckSum);
        updateCheckSum(header, delta);
    }
    
    /**
     * Apply an incremental update to the checksum, as a result of changing one of more records
     * @param header The header to update
     * @param delta The change in the checksum
     */
    public static void updateCheckSum(Header header, long delta) {

        final HeaderCard checkSumCard = header.findCard(CHECKSUM);
        String oldChecksum = checkSumCard.getValue();
        long cksum = (~FitsCheckSum.checksumDecode(oldChecksum)) & FitsIO.INTEGER_MASK;
        cksum += delta;
        // If we had a carry it should go into the beginning.
        while ((cksum & FitsIO.HIGH_INTEGER_MASK) != 0) {
            long cshduIntPart = cksum & FitsIO.INTEGER_MASK;
            cksum = cshduIntPart + (cksum>>32);
        }
        checkSumCard.setValue(FitsCheckSum.checksumEnc(cksum, true));
    }

    /**
     * Add or update the CHECKSUM keyword. by R J Mathar
     * 
     * @param hdu
     *            the HDU to be updated.
     * @throws FitsException
     *             if the operation failed
     * @since 2005-10-05
     */
    public static void setChecksum(BasicHDU<?> hdu) throws FitsException {
        /*
         * the next line with the delete is needed to avoid some unexpected
         * problems with non.tam.fits.Header.checkCard() which otherwise says it
         * expected PCOUNT and found DATE.
         */
        Header hdr = hdu.getHeader();
        hdr.deleteKey(CHECKSUM);
        /*
         * This would need org.nevec.utils.DateUtils compiled before
         * org.nevec.prima.fits .... final String doneAt =
         * DateUtils.dateToISOstring(0) ; We need to save the value of the
         * comment string because this is becoming part of the checksum
         * calculated and needs to be re-inserted again - with the same string -
         * when the second/final call to addValue() is made below.
         */
        hdr.addValue(CHECKSUM, "0000000000000000");

        /*
         * Convert the entire sequence of 2880 byte header cards into a byte
         * array. The main benefit compared to the C implementations is that we
         * do not need to worry about the particular byte order on machines
         * (Linux/VAX/MIPS vs Hp-UX, Sparc...) supposed that the correct
         * implementation is in the write() interface.
         */
        ByteArrayOutputStream hduByteImage = new ByteArrayOutputStream();
        BufferedDataOutputStream bdos = new BufferedDataOutputStream(hduByteImage);

        // DATASUM keyword.
        hdu.getData().write(bdos);
        try {
            bdos.flush();
        } catch (IOException e) {
            log.log(Level.SEVERE, "should not happen", e);
        }
        byte[] data = hduByteImage.toByteArray();
        checksum(data);
        hdu.write(new BufferedDataOutputStream(hduByteImage));
        long csd = checksum(data);
        hdu.getHeader().addValue(DATASUM, Long.toString(csd));

        // We already have the checsum of the data. Lets compute it for
        // the header.
        hduByteImage.reset();
        hdu.getHeader().write(bdos);
        try {
            bdos.flush();
        } catch (IOException e) {
            log.log(Level.SEVERE, "should not happen", e);
        }
        data = hduByteImage.toByteArray();

        long csh = checksum(data);

        long cshdu = csh + csd;
        // If we had a carry it should go into the
        // beginning.
        while ((cshdu & FitsIO.HIGH_INTEGER_MASK) != 0) {
            long cshduIntPart = cshdu & FitsIO.INTEGER_MASK;
            cshdu = cshduIntPart + 1;
        }
        /*
         * This time we do not use a deleteKey() to ensure that the keyword is
         * replaced "in place". Note that the value of the checksum is actually
         * independent to a permutation of the 80-byte records within the
         * header.
         */
        hdr.addValue(CHECKSUM, checksumEnc(cshdu, true));
    }

}
