
package org.lsst.ccs.utilities.ccd.image.data;

import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.utilities.image.ReadOutParameters;


/**
 * Interface for objects that provide DataStreams. 
 * It is via a RawImageDataProvider object that we get the data to be
 * written into fits files.
 * 
 * It provides the ability to add RawImageDataListener for dispatching
 * notifications when data is available.
 * 
 * The methods to support the listener/notification mechanism are static and
 * have a default implementation so that the implementer of RawImageDataProvider
 * has only to implement getRawImageData.
 * 
 * When fetching the RawImageData we provide a ReadOutParameters object that
 * has the parameters defining the area to be read.
 * 
 * @author The LSST CCS Team.
 */
public interface RawImageDataProvider {
    
    /**
     * Get the RawImageData for this object.
     * 
     * @param readOutParameters The ReadOutParameters for the acquisition.
     * @return The RawImageData containing the data.
     */
    public RawImageData getRawImageData(ReadOutParameters readOutParameters);
    
    /**
     * Default static method and fields to support the listener/notification
     * mechanism for classes implementing this interface.
     */

    static List<InnerRawImageDataListener> LISTENERS = new ArrayList<>();
        
    /**
     * Add a RawImageDataListener for a given RawImagedataProvider.
     * 
     * @param listener The listener to add
     * @param provider
     */
    static void addRawImageDataListener(RawImageDataListener listener, RawImageDataProvider provider) {
        LISTENERS.add(new InnerRawImageDataListener(listener,provider));
    }
    
    /**
     * Remove a RawImageDataListener to this RawImageDataProvider.
     * 
     * @param listener The listener to remove
     */
    static void removeRawImageDataListener(RawImageDataListener listener) {
        List<InnerRawImageDataListener> toRemove = new ArrayList<>();
        for ( InnerRawImageDataListener inner : LISTENERS ) {
            if ( inner.getListener() == listener ) {
                toRemove.add(inner);
            }
        }
        for ( InnerRawImageDataListener inner : toRemove ) {
            LISTENERS.remove(inner);            
        }
    }
    
    /**
     * Notify the LISTENERS that RawImageData is available for a given RawImageDataProvider.
     * 
     * @param dataProvider the RawImageDataProvider for which the data is available.
     */
    static void notifyRawImageDataListeners(RawImageDataProvider dataProvider) {
        for ( InnerRawImageDataListener l : LISTENERS ) {
            if ( l.getProvider() == dataProvider ) {
                l.getListener().rawImageDataAvailable(dataProvider);
            }
        }
    }
    
    static class InnerRawImageDataListener {
        final RawImageDataListener listener;
        final RawImageDataProvider provider;
        
        InnerRawImageDataListener(RawImageDataListener listener, RawImageDataProvider provider) {
            this.listener = listener;
            this.provider = provider;
        }
        
        RawImageDataListener getListener() {
            return listener;
        }
        
        RawImageDataProvider getProvider() {
            return provider;
        }
    }
}
