
package org.lsst.ccs.utilities.image.patterns;

import java.awt.image.BufferedImage;
import java.awt.image.WritableRaster;
import java.util.Arrays;
import java.util.Map;

/**
 * A PatternGenerator that creates a random noise in a pixel block pattern 
 * varying from a baseline to a maximum amplitude in gray scale.
 * 
 * The allowed parameters for this patternGenerator are:
 *   - noise_width: the width of the noise block in pixels
 *   - noise_ySize: the height of the noise block in pixels
 *   - noise_maxAmplitude: the maximum amplitude of the noise. An integer from 0 to 256
 *   - noise_baseline: the noise baseline. An integer form 0 to 256
 *   - ripples_frequency: the ripples frequency
 * 
 * The idea for this code was taken from http://www.lac.inpe.br/JIPCookbook/1200-create-gl.jsp
 * 
 * @author turri
 */
public class NoisePatternGenerator extends PatternGenerator {
    
    NoisePatternGenerator() {
        super("noise");
    }

    @Override
    public GeneratedImage generateImage(int width, int height, Map<String, String> parameters) {

        int noiseWidthSize = getIntParameter("noise_xSize", parameters, 15);
        int noiseHightSize = getIntParameter("noise_ySize", parameters, 15);
        
        int nx = width/noiseWidthSize + 1;
        int ny = height/noiseHightSize + 1;
        
        int totalNoiseWidth = nx*noiseWidthSize;
        int totalNoiseHeight = ny*noiseHightSize;

        int noise_baseline = getIntParameter("noise_baseline", parameters, 0);
        int noise_maxAmplitude = getIntParameter("noise_maxAmplitude", parameters, 256);
        
        int maxFluctuation = Math.max(noise_maxAmplitude - noise_baseline,0);
        

        WritableRaster raster = createRaster(totalNoiseWidth, totalNoiseHeight);
        for (int h = 0; h < ny; h++) {
            for (int w = 0; w < nx; w++) {
                int[] fill = new int[noiseWidthSize*noiseHightSize]; // A block of pixels...
                Arrays.fill(fill,(int)(noise_baseline + Math.random()*maxFluctuation));  // .. filled a random gray level.
                raster.setSamples(w * noiseWidthSize, h * noiseHightSize, noiseWidthSize, noiseHightSize, 0, fill);
         }
        }
        
        int[][]data = new int[width][height];
        for (int i = 0; i < width; i++) {
            for (int j = 0; j < height; j++) {
                data[i][j] = (int) raster.getSample(i, j, 0);
            }
        }
        
        
        return new GeneratedImage( data, noise_maxAmplitude );
    }
    
}
