package org.lsst.ccs.utilities.image;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import nom.tam.fits.FitsException;

/**
 * A specification for a fits file header. Typically read from a .spec file.
 * For a given HeaderSpecification it is possible to read multiple spec files
 * to allow for overwriting of information on the part of subsystems.
 * 
 * Any uploaded spec file will overwrite the existing information.
 * 
 * @author tonyj
 */
public class HeaderSpecification {

    private String name;
    private final Map<String, HeaderLine> headers = new LinkedHashMap<>();
    private final static Pattern EXPRESSION_PATTERN = Pattern.compile("\\$\\{(.*)}");
    private final static Pattern LINE_PATTERN = Pattern.compile("\\s*(\\S+)\\s+(\\w+)\\s+(?:\"(.*)\"|(\\S+))\\s*(.*)");

    public enum DataType {
        Boolean, Integer, String, Float, Date, MJD
    };

    public HeaderSpecification(String name, InputStream in) throws IOException {
        this(name, new InputStreamReader(in));
    }

    public HeaderSpecification(String name, Reader in) throws IOException {
        this.name = name;
        loadHeaderSpecification(in);
    }

    /**
     * Load Header Specification information from an InputStream.
     * The content of the HeaderSpecification will be updated based on the 
     * provided information.
     * Existing information will be over-written so that the end result will
     * be the superposition of what already exists with what has been provided.
     * 
     * @param in The input stream with the HeadSpecification information.
     * @throws java.io.IOException For any problems with the InputStream.
     */    
    final void loadHeaderSpecification(InputStream in) throws IOException {
        loadHeaderSpecification(new InputStreamReader(in));
        
    }

    final void loadHeaderSpecification(Reader in) throws IOException {
        BufferedReader reader = in instanceof BufferedReader
                ? (BufferedReader) in : new BufferedReader(in);
        for (;;) {
            String line = reader.readLine();
            if (line == null) {
                break;
            } else if (line.startsWith("#")) {
                continue;
            }
            try {
                HeaderLine headerLine = new HeaderLine(line);
                headers.put(headerLine.getKeyword(), headerLine);
            } catch (IOException x) {
                throw new IOException("Error while reading line:\n\t" + line, x);
            }
        }        
    }
    
    /**
     * The name of the HeaderSpecification. It is used to map the HeaderSpecification
     * with the fits header being written out.
     * 
     * @return 
     */
    public String getName() {
        return name;
    }

    /**
     * Get a specific HeaderLine object specifying the full definition for
     * a given header keyword.
     * 
     * @param keyWord The keyword name for the HeaderLine
     * @return        The HeaderLine defining the content of the header.
     */
    public HeaderLine getHeader(String keyWord) {
        return headers.get(keyWord);
    }
    
    /**
     * Get the full collection of HeaderLines for this HeaderSpecification.
     * The returned collection is unmodifiable.
     * @return The unmodifiable collection of HeaderLine objects.
     */
    public Collection<HeaderLine> getHeaders() {
        return Collections.unmodifiableCollection(headers.values());
    }

    /**
     * A Class defining each Header line contained in a HeaderSpecification.
     * A HeaderLine is built from each of the lines contained in a header specification
     * file (.spec file).
     */
    public static class HeaderLine {

        private final String keyword;
        private final DataType dataType;
        private final String comment;
        private final String metaName;
        private final Object value;
        private final String metaMap;
        private final boolean isExpression;
        //For test only: https://jira.slac.stanford.edu/browse/LSSTCCS-1027
        //To-be removed when the Warning is removed
        private final boolean isWarning;

        HeaderLine(String line) throws IOException {
            try {
                Matcher lineMatcher = LINE_PATTERN.matcher(line);
                if (!lineMatcher.matches()) {
                    throw new IOException("Invalid line in header specification");
                }
                keyword = lineMatcher.group(1);
                dataType = DataType.valueOf(lineMatcher.group(2));
                String valueExpression = lineMatcher.group(3) == null ? lineMatcher.group(4) : lineMatcher.group(3);
                Matcher matcher = EXPRESSION_PATTERN.matcher(valueExpression);
                isExpression = matcher.matches();
                if (isExpression) {
                    String expression = matcher.group(1);
                    int dotIndex = expression.indexOf(".");
                    int slashIndex = expression.indexOf("/");
                    if ( dotIndex == -1 || (dotIndex != -1 && slashIndex != -1 && slashIndex < dotIndex )) {
                        metaMap = null;
                        metaName = expression;
                    } else {
                        metaMap = expression.substring(0,dotIndex);
                        metaName = expression.substring(dotIndex +1);
                    }
                    value = null;
                    isWarning = (metaMap == null ||  metaMap.isEmpty() ) && metaName.contains("/");
                    if ( isWarning ) {
                        System.out.println("WARNING: path like format to identify trending quantities has been discontinued.");
                        System.out.println("WARNING: use "+metaName.replaceFirst("/", ".")+" instead of "+metaName);
                    }                    
                } else {
                    metaMap = null;
                    metaName = null;
                    value = coerce(valueExpression,dataType);
                    isWarning = false;
                }
                comment = lineMatcher.group(5);
            } catch (IllegalArgumentException | FitsException x) {
                throw new IOException("Illegal token found while reading header specification", x);
            }
        }

        /**
         * The keyword name as it will be written in the fits file.
         * 
         * @return The keyword name.
         */
        public String getKeyword() {
            return keyword;
        }

        /**
         * The data type for this HeaderLine.
         * 
         * @return The data type.
         */
        public DataType getDataType() {
            return dataType;
        }

        public String getMetaName() {
            return metaName;
        }
        
        String getMetaMap() {
            return metaMap;
        }
        /**
         * The Comment as it will be written in the fits file.
         * It describes the keyword.
         * 
         * @return The comment line.
         */
        public String getComment() {
            return comment;
        }
        
        Object getValue(MetaDataSet metaDataSet) {
            if (isExpression) {
                return findMetaDataValue(metaMap, metaName, metaDataSet);
            } else {
                return value;
            }
        }

        boolean isWarning() {
            return isWarning;
        }
        
        private Object findMetaDataValue(String map, String name, MetaDataSet metaDataSet) {
            return metaDataSet.getValue(map, name);
        }

        private Object coerce(String valueExpression, DataType dataType) throws FitsException, NumberFormatException {
            switch (dataType) {
                case Integer: return Integer.decode(valueExpression);
                case Float: return Double.valueOf(valueExpression);
                case Boolean: return Boolean.valueOf(valueExpression);
                case Date: return DateUtils.convertStringToDate(valueExpression);
                case MJD: return Double.valueOf(valueExpression);
                default: return valueExpression;
            }
        }
    }
}
