package org.lsst.ccs.utilities.ccd;

import java.awt.Dimension;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.BitSet;
import java.util.List;
import org.lsst.ccs.utilities.ccd.image.data.RawImageData;
import org.lsst.ccs.utilities.ccd.image.data.RawImageDataProvider;
import org.lsst.ccs.utilities.image.ReadOutParameters;

/**
 * A Reb Geometry object. A Reb contains 3 CCDs of the same type.
 *
 * A Reb implements RawImageDataProvider, as it can be used as a source of
 * raw pixel data.
 *
 * @author The LSST CCS Team.
 */
public class Reb extends Geometry<CCD> implements RawImageDataProvider {

    private final int gap_iny;
    private final static RawImageData.BitsPerPixel bits = RawImageData.BitsPerPixel.BIT32;
    private final RaftGeometryConstants raftConstants;

    /**
     * Create an empty Reb instance by providing RaftGeometryConstants.
     * Funny to use Geometry Constants but that's the way it is implemented for
     * the moment.
     *
     * @param raftGeometryConstants The Raft Geometry constants
     */
    Reb(RaftGeometryConstants raftGeometryConstants) {
        super("Reb", new Dimension(raftGeometryConstants.getPhysicalAreaParallelSize(), raftGeometryConstants.getRafty() - (int) (2 * raftGeometryConstants.getGap_outy())), 1, 3);
        this.gap_iny = raftGeometryConstants.getGap_iny();
        this.raftConstants = raftGeometryConstants;
    }

    /**
     * Create an empty Reb instance for a given CCD type.
     *
     * @param type The type of the CCD
     */
    Reb(CCDType type) {
        this(RaftGeometryConstants.createRaftGeometryConstants(type));
    }

    @Override
    public String getName() {
        return name + getParallelPosition();
    }

    @Override
    protected void addGeometryToGrid(CCD child, int p, int s) {
        if (p != 0) {
            throw new IllegalArgumentException("Rebs only allow one row of CCDs in it");
        }
        int yCoord = (getSerialChildrenCount() - 1 - s) * (child.getHeight() + gap_iny);
        addGeometry(child, 0, yCoord);
    }

    @Override
    protected void setParallelPosition(int parallelPosition) {
        super.setParallelPosition(parallelPosition);
        for (CCD ccd : getChildrenList()) {
            ccd.setParallelPosition(parallelPosition);
        }
    }

    /**
     * Utility method to create a Reb populated with the specified CCDs for
     * the given type.
     *
     * @param type The type of CCD
     * @param ccdBitSet BitSet representing which CCDs are populating this Reb
     * @return A Reb containing the specified CCDs.
     */
    public static Reb createReb(CCDType type, BitSet ccdBitSet) {
        Reb reb = new Reb(type);
        //Add three CCDs left to right
        for (int i = 0; i < 3; i++) {
            if (ccdBitSet.get(i)) {
                CCD ccd = CCD.createCCD(type);
                reb.addChildGeometry(ccd, 0, i);
            }
        }
        return reb;
    }

    /**
     * Utility method to create a fully populated Reb for a given CCD type.
     *
     * @param type The type of CCD
     * @return A fully populated Reb.
     */
    public static Reb createReb(CCDType type) {
        BitSet s = new BitSet(3);
        s.set(0, 3);
        return createReb(type, s);
    }

    protected RaftGeometryConstants getRaftGeometryConstants() {
        return raftConstants;
    }

    //TO DO: Do we need to copy the data there?
    //Please notice that the format for the output data is
    //in buckets of corresponding segment pixels, i.e.
    //the format the DAQ is providing. This should NOT be
    //done here, but somewhere else where the DAQ is simulated.
    @Override
    public RawImageData getRawImageData(ReadOutParameters readOutParameters) {
        int totalSize = 0;
        List<ByteBuffer> availableData = new ArrayList<>();
        int segCount = 0;
        for (int s = 0; s < getSerialChildrenCount(); s++) {
            CCD ccd = getChild(0, s);
            for (Segment segment : ccd.getSegments()) {
                RawImageData imageData = ((Segment) segment).getRawImageData(readOutParameters);
                availableData.add(imageData.getImageData());
                totalSize += imageData.getImageData().limit();
                segCount++;
            }
        }

        ByteBuffer result = ByteBuffer.allocateDirect(totalSize);
        result.order(ByteOrder.nativeOrder());
        int segmentData = totalSize / segCount / bits.bytes();
        for (int i = 0; i < segmentData; i++) {
            for (ByteBuffer byteBuffer : availableData) {
                result.putInt(byteBuffer.getInt());
            }
        }

        result.flip();
        return new RawImageData(bits, result);
    }

}
