package org.lsst.ccs.utilities.image.patterns;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import org.lsst.ccs.utilities.ccd.Geometry;
import org.lsst.ccs.utilities.ccd.ImageSensitiveArea;
import org.lsst.ccs.utilities.ccd.image.data.RawImageDataProvider;

/**
 * A factory to generate patterns.
 *
 * @author turri
 */
public class PatternGeneratorFactory {

    private final static HashMap<String, Class> patterns = new HashMap<>();

    static {
        registerPatternGenerator(RipplesPatternGenerator.class);
        registerPatternGenerator(NoisePatternGenerator.class);
        registerPatternGenerator(CheckeredPatternGenerator.class);
        registerPatternGenerator(ImagePatternGenerator.class);
    }

    
    /**
     * List the patterns that are available for generating the Image.
     * 
     * @return The Set of the available patterns.
     */
    public static Set<String> listAvailablePatterns() {
        return patterns.keySet();
    }

    
    /**
     * Creates a new PatternGenerator for the given name.
     * This method returns a new instance of a PatternGenerator so that different
     * instances can be used on separate threads.
     * A RuntimeException will be thrown if an instance of the given PatternGenerator
     * cannot be instanced.
     *
     * @param name The name of the PatternGenerator.
     * @return The desired PatternGenerator.
     *
     */
    public static PatternGenerator createPatternGenerator(String name) {
        if (!patterns.containsKey(name)) {
            throw new RuntimeException("Pattern \"" + name + "\" does not exist");
        }
        try {
            return (PatternGenerator) patterns.get(name).newInstance();
        } catch (IllegalAccessException | InstantiationException e) {
            throw new RuntimeException("Could not create PatterGenerator for \"" + name + "\".", e);
        }
    }

    /**
     * Register a new PatternGenerator by providing its Class.
     * This method will throw a IllegalArgumentException if the provided class
     * does not have an empty constructor. In such case the PatternGenerator
     * will not be registered.
     * An IllegalArgumentException will also be thrown if a PatternGenerator
     * with the same "name" exists @see{PatternGenerator#getName()}.
     *
     * @param patternGeneratorClass The Class of the PatternGenerator.
     *
     */
    private static void registerPatternGenerator(Class patternGeneratorClass) {
        try {
            PatternGenerator instance = (PatternGenerator) patternGeneratorClass.newInstance();
            String name = instance.getName();
            if (patterns.containsKey(name)) {
                throw new IllegalArgumentException("PatternGenerator with name \"" + name + "\" is already registered");
            }
            patterns.put(name, patternGeneratorClass);
        } catch (IllegalAccessException | InstantiationException e) {
            throw new IllegalArgumentException("Could not register the pattern generator class " + patternGeneratorClass.getCanonicalName(), e);
        }
    }    
    
    /**
     * Generate a simulated image for the given Geometry.
     * @param geometry    The Geometry for which to simulate an image
     * @param imagePattern The simulated image pattern
     * @param params       The parameters for the simulated image.
     * @return 
     */
    public static GeneratedImage generateImageForGeometry(Geometry<?> geometry, String imagePattern, Map<String,String> params) {
        if (geometry == null) {
            throw new RuntimeException("The Geometry cannot be null");
        }        
        int width = geometry.getWidth();
        int height = geometry.getHeight();

        System.out.print("Generating image " + imagePattern + " " + width + "x" + height+" took ");
        long start = System.currentTimeMillis();

        PatternGenerator patternGenerator = PatternGeneratorFactory.createPatternGenerator(imagePattern);
        GeneratedImage img =  patternGenerator.generateImage(width, height, params);
        int seconds = (int)((System.currentTimeMillis() - start)/1000.);
        System.out.println(seconds+" seconds.");
        return img;
    }
    
    /**
     * Expose a Geometry object to a generated image.
     * Internally the object's geometry is scanned to look for objects that can
     * be exposed to images (ImageSensitiveArea).
     * 
     * @param geometry The Geometry to be exposed 
     * @param generatedImage
     */
    public static void exposeGeometryToGeneratedImage(Geometry<?> geometry, GeneratedImage generatedImage) {
        if ( geometry instanceof ImageSensitiveArea ) {
            ((ImageSensitiveArea) geometry).exposeToImage(generatedImage);
            
        }
        for ( Geometry child : geometry.getChildrenList() ) {
            exposeGeometryToGeneratedImage(child, generatedImage);
        }
        if ( geometry instanceof RawImageDataProvider ) {
            RawImageDataProvider.notifyRawImageDataListeners((RawImageDataProvider)geometry);
        }
    }
    
}
