package org.lsst.ccs.utilities.image.demo;

import java.io.File;
import java.io.IOException;
import java.util.List;
import java.util.concurrent.FutureTask;
import nom.tam.fits.FitsException;
import org.lsst.ccs.utilities.ccd.CCD;
import org.lsst.ccs.utilities.ccd.CCDType;
import org.lsst.ccs.utilities.ccd.E2VCCDType;
import org.lsst.ccs.utilities.ccd.Raft;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.ccd.Segment;
import org.lsst.ccs.utilities.image.FitsHeaderUtilities;
import org.lsst.ccs.utilities.image.FitsFileWriter;
import org.lsst.ccs.utilities.image.FitsHeadersSpecificationsBuilder;
import org.lsst.ccs.utilities.image.ImageSet;
import org.lsst.ccs.utilities.image.ReadOutParameters;
import org.lsst.ccs.utilities.image.patterns.GeneratedImage;
import org.lsst.ccs.utilities.image.patterns.PatternGeneratorFactory;

/**
 * TEST CLASS FOR PARALLEL/SEQUENTIAL Fits file writing.
 * 
 * This test writes fits files in series and in parallel and it checks
 * that the content is the same.
 * 
 * @author turri
 */
public class RaftsFitsWritingDemo {

    private static final String fitsFileName = "/tmp/R99_S%d%d";

    public static void main(String[] argv) throws IOException, FitsException, InterruptedException {

        FitsHeadersSpecificationsBuilder headerSpecsBuilder = new FitsHeadersSpecificationsBuilder();        
        headerSpecsBuilder.addSpecFile("primary");
        headerSpecsBuilder.addSpecFile("extended");
        headerSpecsBuilder.addSpecFile("test_cond");

        CCDType type = new E2VCCDType();
        Raft raft = Raft.createRaft(type);

        GeneratedImage generatedImage = PatternGeneratorFactory.generateImageForGeometry(raft, "ripples", null);
        PatternGeneratorFactory.exposeGeometryToGeneratedImage(raft, generatedImage);

        writeFitsFilesForRaft(raft, true, headerSpecsBuilder);
        writeFitsFilesForRaft(raft, false, headerSpecsBuilder);

    }

    private static void writeFitsFilesForRaft(Raft raft, boolean inParallel, FitsHeadersSpecificationsBuilder headerSpecsBuilder) throws IOException, FitsException, InterruptedException {
        String how = inParallel ? "in parallel" : "sequentially";
        System.out.print("Writing out fits files " + how);
        long start = System.currentTimeMillis();

        Thread[] ts = new Thread[raft.getParallelChildrenCount() * 3];

        for (int rebId = 0; rebId < raft.getParallelChildrenCount(); rebId++) {
            Reb reb = raft.getChild(rebId, 0);
            for (int ccdId = 0; ccdId < 3; ccdId++) {
                CCD ccd = reb.getChild(0, ccdId);
                String raftName = String.format(fitsFileName, rebId, ccdId);
                if ( inParallel ) {
                    raftName += "_parallel";
                }
                File raftFile = new File(raftName + ".fits");
                if ( raftFile.exists() ) {
                    raftFile.delete();
                }

                ImageSet imageSet = FitsHeaderUtilities.createImageSetForCCD(ccd);

                FitsFileWriter ffw = new FitsFileWriter(raftFile, imageSet, headerSpecsBuilder.getHeaderSpecifications());
                if (inParallel) {
                    Thread t = new Thread(new FitsFileWriterRunnable(ffw, ccd.getSegments(), imageSet.getReadOutParameters()));
                    ts[rebId * 3 + ccdId] = t;
                    t.start();
                } else {
                    for (Segment segment : ccd.getSegments()) {
                        ffw.write(segment.getChannel() - 1, ((Segment) segment).getRawImageData(imageSet.getReadOutParameters()).getImageData());
                    }
                }

            }
        }

        if ( inParallel ) {
            for (Thread t : ts) {
                t.join();
            }
        }
        double totalTime = (double) (System.currentTimeMillis() - start) / 1000.;
        System.out.println(" took " + totalTime + " seconds");

    }

    private static class FitsFileWriterRunnable implements Runnable {

        private final FitsFileWriter ffw;
        private final List<Segment> segments;
        private final ReadOutParameters readOutParameters;

        FitsFileWriterRunnable(FitsFileWriter ffw, List<Segment> segments, ReadOutParameters readOutParameters) {
            this.ffw = ffw;
            this.segments = segments;
            this.readOutParameters = readOutParameters;
        }

        @Override
        public void run() {
            try {

            FutureTask<Void>[] segmentThreads = new FutureTask[segments.size()];

            for (Segment segment : segments) {
                segmentThreads[segment.getChannel()-1] = ffw.asyncWrite(segment.getChannel()-1, ((Segment) segment).getRawImageData(readOutParameters).getImageData());
            }
            
            for (Segment segment : segments) {
                    segmentThreads[segment.getChannel()-1].get();
            }
            
            
            } catch (Exception ioe) {
                throw new RuntimeException(ioe);
            }
        }

    }

}
