package org.lsst.ccs.utilities.image;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.utilities.ccd.CCD;
import org.lsst.ccs.utilities.ccd.CCDType;
import org.lsst.ccs.utilities.ccd.E2VCCDType;
import org.lsst.ccs.utilities.ccd.Geometry;
import org.lsst.ccs.utilities.ccd.Raft;
import org.lsst.ccs.utilities.ccd.Segment;

/**
 * Utility class for the generation of header specifications objects.
 *
 * @author turri
 */
public class FitsHeaderUtilities {

    /**
     * Utility function to return a pair of values in Fits standard.
     *
     * @param a Lower edge
     * @param b Upper edge
     * @return Returns the par of values in Fits Standard.
     *
     */
    private static String pair(String a, String b) {
        return "[" + a + "," + b + "]";
    }

    /**
     * Utility function to return a range of values in Fits standard.
     *
     * @param a Lower edge
     * @param b Upper edge
     * @return Returns the range in Fits Standard.
     *
     */
    private static String range(int a, int b) {
        return a + ":" + b;
    }

    /**
     * Create the default ImageSet for the given CCD.
     * 
     * @param ccd The CCD for which to create the ImageSet
     * @return The corresponding ImageSet
     * 
     * TO-DO ReadOutParameters must be passed in, rather than created.
     * 
     */
    public static ImageSet createImageSetForCCD(CCD ccd) {
        CCDType type = ccd.getType();
        ReadOutParameters readOutParameters = new ReadOutParameters(type);
        DefaultImageSet imageSet = new DefaultImageSet(ccd, readOutParameters);        
        return imageSet;
    }
    
    
    /**
     * Create fits headers for the primary header for the given Object.
     *
     * @param ccd The CCD for which the primary header is returned.
     * @return A map containing the fits header names and values.
     */
    public static Map<String, Object> getCCDPrimaryHeaders(CCD ccd) {
        Map<String, Object> primaryMetaData = new HashMap<>();
        primaryMetaData.put("DETSIZE", getDETSIZE(ccd.getType()) );
        primaryMetaData.put("CCDSlot", ccd.getName());
        Geometry reb = ccd.getParent();
        if ( reb != null ) {
            Geometry raft = reb.getParent();
            if ( raft != null ) {
                primaryMetaData.put("RAFTBay", raft.getName());                
            }
        }
        return primaryMetaData;
    }


    /**
     * Create fits headers for a Segment given the provided ReadOutParameters. 
     * Optionally create distorted headers so that over and pre scan regions 
     * can be viewed in DS9, as requested by users at BNL.
     *
     * @param segment The segment for which to build the Header.
     * @param readOutParameters The ReadOutParameters used to acquire the image.
     * @return A map containing the fits header names and values.
     */
    public static Map<String, Object> getSegmentHeaders(Segment segment, ReadOutParameters readOutParameters) {

        Map<String, Object> imageMetaData = new HashMap<>();

        imageMetaData.put("NAXIS", 2);
        imageMetaData.put("NAXIS1", readOutParameters.getSerialReadPixels());
        imageMetaData.put("NAXIS2", readOutParameters.getParallelReadPixels());
        
        imageMetaData.put("EXTNAME", String.format("Segment%01d%01d", getSegmentParallelPosition(segment), getSegmentSerialPosition(segment)));
        imageMetaData.put("CHANNEL", segment.getChannel());
        imageMetaData.put("CCDSUM", "1 1");

//        imageMetaData.put("CUNIT1", "pixel");
//        imageMetaData.put("CUNIT2", "pixel");


        CCD ccd = (CCD) segment.findParentOfType(CCD.class);
        CCDType type = ccd.getType();
        boolean isE2V = type instanceof E2VCCDType;

        //Mosaic Keywords
        imageMetaData.put("DTM1_1", isE2V ? getOneMinusTwoSx(segment) : -1);
        imageMetaData.put("DTM2_2", -1*getOneMinusTwoSx(segment));
        imageMetaData.put("DTV1", getDTV1(segment,readOutParameters));
        imageMetaData.put("DTV2", getDTV2(segment));
        imageMetaData.put("DETSIZE", getDETSIZE(type) );
        imageMetaData.put("DATASEC", pair(range(1+readOutParameters.getSerialPrescan(), readOutParameters.getSerialPrescan()+type.getCCDGeometryConstants().getSegmentSerialActiveSize()), range(1, type.getCCDGeometryConstants().getSegmentParallelActiveSize())));
        imageMetaData.put("DETSEC", getDETSEC(segment));
        

        //WCS keywords
        //Amplifier
        imageMetaData.put("PC1_2A", getOneMinusTwoSx(segment));
        imageMetaData.put("PC2_1A", isE2V ? getOneMinusTwoSx(segment) : -1);
        //CCD
        imageMetaData.put("PC1_2C", getOneMinusTwoSx(segment));
        imageMetaData.put("PC2_1C", isE2V ? getOneMinusTwoSx(segment) : -1);
        //Raft
        imageMetaData.put("PC1_2R", getOneMinusTwoSx(segment));
        imageMetaData.put("PC2_1R", isE2V ? getOneMinusTwoSx(segment) : -1);
        //Focal Plane
        imageMetaData.put("PC1_2F", getOneMinusTwoSx(segment));
        imageMetaData.put("PC2_1F", isE2V ? getOneMinusTwoSx(segment) : -1);
        //CCD Serial/Parallel
        imageMetaData.put("PC1_1B", isE2V ? getOneMinusTwoSx(segment) : -1);
        imageMetaData.put("PC2_2B", getOneMinusTwoSx(segment));
        //Raft Serial/Parallel
        imageMetaData.put("PC1_1Q", isE2V ? getOneMinusTwoSx(segment) : -1);
        imageMetaData.put("PC2_2Q", getOneMinusTwoSx(segment));

        //WCS CRVAL keywords
        //Amplifier
        imageMetaData.put("CRVAL1A", getCRVAL1A(segment));
        imageMetaData.put("CRVAL2A", getCRVAL2A(segment,readOutParameters));
        //CCD
        imageMetaData.put("CRVAL1C", getCRVAL1C(segment));
        imageMetaData.put("CRVAL2C", getCRVAL2C(segment,readOutParameters));
        //CCD Serial/Parallel
        imageMetaData.put("CRVAL1B", getCRVAL2C(segment,readOutParameters));
        imageMetaData.put("CRVAL2B", getCRVAL1C(segment));

        if ( ccd != null ) {
            Raft raft = (Raft)ccd.findParentOfType(Raft.class);
            if ( raft != null ) {
            //Raft
            imageMetaData.put("CRVAL1R", getCRVAL1R(segment,ccd,raft));
            imageMetaData.put("CRVAL2R", getCRVAL2R(segment,ccd,raft,readOutParameters));
            //Raft Serial/Parallel
            imageMetaData.put("CRVAL1Q", getCRVAL2R(segment,ccd,raft,readOutParameters));
            imageMetaData.put("CRVAL2Q", getCRVAL1R(segment,ccd,raft));
                //Focal Plane
                imageMetaData.put("CRVAL1F", getCRVAL1F(segment,ccd,raft));
                imageMetaData.put("CRVAL2F", getCRVAL2F(segment,ccd,raft,readOutParameters));
            }
        }

        return imageMetaData;
    }
    
    private static int getSegmentParallelPosition(Segment seg) {
        return seg.getParallelPosition();
    }
    private static int getSegmentSerialPosition(Segment seg) {
        return seg.getSerialPosition();
    }
    
    private static boolean isReadoutDown(Segment seg) {
        return seg.isReadoutDown();            
    }
    private static boolean isReadoutLeft(Segment seg) {
        return seg.isReadoutLeft();
    }

    /**
     * Returns 1 - 2*Sx 
     * @param seg
     * @return 
     */
    private static double getOneMinusTwoSx(Segment seg) {
        return 1 - 2*seg.getParallelPosition();
    }

    private static String getDETSIZE(CCDType type) {
        return pair(range(1, 8*type.getCCDGeometryConstants().getSegmentSerialActiveSize()), range(1, 2*type.getCCDGeometryConstants().getSegmentParallelActiveSize()));
    }
    
    private static double getCRVAL1A(Segment seg) {
        return seg.getParallelPosition()*(((CCD)seg.findParentOfType(CCD.class)).getType().getCCDGeometryConstants().getSegmentParallelActiveSize()+1);        
    }
    
    private static double getCRVAL2A(Segment seg, ReadOutParameters readOutParameters) {
        CCDType type = ((CCD)seg.findParentOfType(CCD.class)).getType();
        double sx = seg.getParallelPosition(); 
        if ( type instanceof E2VCCDType ) {
            return  sx*(type.getCCDGeometryConstants().getSegmentSerialActiveSize()+1) + (2*sx - 1)*readOutParameters.getSerialPrescan();        
        } else {
            return type.getCCDGeometryConstants().getSegmentSerialActiveSize() + 1 - readOutParameters.getSerialPrescan();
        }
    }

    private static double getCRVAL1C(Segment seg) {
        return seg.getParallelPosition()*(2*((CCD)seg.findParentOfType(CCD.class)).getType().getCCDGeometryConstants().getSegmentParallelActiveSize()+1);        
    }
    
    private static double getCRVAL2C(Segment seg,ReadOutParameters readOutParameters) {
        CCDType type = ((CCD)seg.findParentOfType(CCD.class)).getType();
        double sy = seg.getSerialPosition(); 
        return getCRVAL2A(seg,readOutParameters) + sy*type.getCCDGeometryConstants().getSegmentSerialActiveSize();        
    }

    private static double getCRVAL1R(Segment seg, CCD ccd, Raft raft) {
        CCDType type = ((CCD)seg.findParentOfType(CCD.class)).getType();
        return getCRVAL1C(seg) + raft.getRaftGeometryConstants().getGap_outx() + 0.5*(type.getCCDGeometryConstants().getPhysicalAreaParallelSize()-type.getCCDGeometryConstants().getActiveAreaParallelSize()) + ccd.getParallelPosition()*
                (2*type.getCCDGeometryConstants().getSegmentParallelActiveSize()+raft.getRaftGeometryConstants().getGap_inx()+type.getCCDGeometryConstants().getPhysicalAreaParallelSize()-type.getCCDGeometryConstants().getActiveAreaParallelSize());        
    }
    
    private static double getCRVAL2R(Segment seg, CCD ccd, Raft raft,ReadOutParameters readOutParameters) {
        CCDType type = ((CCD)seg.findParentOfType(CCD.class)).getType();
        return getCRVAL2C(seg,readOutParameters) + raft.getRaftGeometryConstants().getGap_outy() + 0.5*(type.getCCDGeometryConstants().getPhysicalAreaSerialSize()-type.getCCDGeometryConstants().getActiveAreaSerialSize()) + ccd.getSerialPosition()*
                (8*type.getCCDGeometryConstants().getSegmentSerialActiveSize()+raft.getRaftGeometryConstants().getGap_iny()+type.getCCDGeometryConstants().getPhysicalAreaSerialSize()-type.getCCDGeometryConstants().getActiveAreaSerialSize());        
    }

    private static double getCRVAL1F(Segment seg, CCD ccd, Raft raft) {
        CCDType type = ((CCD)seg.findParentOfType(CCD.class)).getType();
        return getCRVAL1R(seg,ccd, raft) + raft.getParallelPosition()*raft.getRaftGeometryConstants().getRaftx();        
    }
    
    private static double getCRVAL2F(Segment seg, CCD ccd, Raft raft, ReadOutParameters readOutParameters) {
        CCDType type = ((CCD)seg.findParentOfType(CCD.class)).getType();
        return getCRVAL2R(seg,ccd, raft,readOutParameters) + raft.getSerialPosition()*raft.getRaftGeometryConstants().getRafty();        
    }

    private static int getDTV1(Segment seg, ReadOutParameters readOutParameters) {
        CCDType type = ((CCD)seg.findParentOfType(CCD.class)).getType();
        int sx = type instanceof E2VCCDType ? seg.getParallelPosition() : 1;        
        return (type.getCCDGeometryConstants().getSegmentSerialActiveSize() + 1 + 2*readOutParameters.getSerialPrescan())*sx+seg.getSerialPosition()*type.getCCDGeometryConstants().getSegmentSerialActiveSize()-readOutParameters.getSerialPrescan();        
    }

    private static int getDTV2(Segment seg) {
        CCDType type = ((CCD)seg.findParentOfType(CCD.class)).getType();
        return (1-seg.getParallelPosition())*(2*type.getCCDGeometryConstants().getSegmentParallelActiveSize()+1);        
    }

    private static int getDSX1(Segment seg) {
        CCDType type = ((CCD)seg.findParentOfType(CCD.class)).getType();
        int sx = seg.getParallelPosition();
        int sy = seg.getSerialPosition();
        if ( type instanceof E2VCCDType ) {
            int res = (sy+1)*type.getCCDGeometryConstants().getSegmentSerialActiveSize()*sx;
            return res + (sy*type.getCCDGeometryConstants().getSegmentSerialActiveSize()+1)*(1-sx);
        } else {
            return (sy+1)*type.getCCDGeometryConstants().getSegmentSerialActiveSize();
        }
    }

    private static int getDSX2(Segment seg) {
        CCDType type = ((CCD)seg.findParentOfType(CCD.class)).getType();
        int sx = seg.getParallelPosition();
        int sy = seg.getSerialPosition();
        if ( type instanceof E2VCCDType ) {
            int res = (sy*type.getCCDGeometryConstants().getSegmentSerialActiveSize()+1)*sx;
            return res + (sy+1)*type.getCCDGeometryConstants().getSegmentSerialActiveSize()*(1-sx);
        } else {
            return sy*type.getCCDGeometryConstants().getSegmentSerialActiveSize()+1;
        }
    }
    
    private static int getDSY1(Segment seg) {
        CCDType type = ((CCD)seg.findParentOfType(CCD.class)).getType();
        int sx = seg.getParallelPosition();
        return 2*type.getCCDGeometryConstants().getSegmentParallelActiveSize()*(1-sx)+sx;
    }
    
    private static int getDSY2(Segment seg) {
        CCDType type = ((CCD)seg.findParentOfType(CCD.class)).getType();
        int sx = seg.getParallelPosition();
        return (type.getCCDGeometryConstants().getSegmentParallelActiveSize()+1)*(1-sx)+type.getCCDGeometryConstants().getSegmentParallelActiveSize()*sx;
    }
    
    
    private static String getDETSEC(Segment segment) {
        return pair(range(getDSX1(segment),getDSX2(segment)), range(getDSY1(segment), getDSY2(segment)));
    }
    
}
