package org.lsst.ccs.utilities.ccd;

import org.lsst.ccs.utilities.image.ReadOutParameters;

/**
 * Allows transforms between CCD coordinate system and CCD segment (amplifier) 
 * coordinate systems. The CCD coordinate system uses the conventions of the camera coordinate
 * system (confusingly called CCS).
 * Any set method may be used to set the currently selected coordinate, and all get 
 * methods will then return coordinates corresponding to the selected coordinate. The
 * precise meaning of the x,y coordinates depends on which method was used to obtain
 * the transform.
 * @see CCDGeometry#getGlobalTransform() 
 * @see CCDGeometry#getActiveTransform() 
 * @author tonyj
 */
public interface CCDTransform {

    /**
     * Enumerate the different types of pixel that can be read out.
     */
    public enum PixelType {
        /**
         * An active (signal) pixel 
         */
        ACTIVE, SERIAL_PRESCAN, SERIAL_OVERSCAN, PARALLEL_OVERSCAN}
    /**
     * Select the absolute x,y CCD coordinate.
     * @param x The x coordinate in the CCD coordinate system
     * @param y The u coordinate in the CCD coordinate system
     */
    void setXY(int x, int y);
    /**
     * Get the CCD x coordinate
     * @return The selected CCD x coordinate
     */
    int getX();
    /**
     * Get the CCD selected y coordinate system
     * @return The CCD y coordinate
     */
    int getY();
    /**
     * Set the current segment, and serial and parallel pixel position within
     * the segment (in segment readout order).
     * @param segment The CCD segment (amplifier)
     * @param serial The serial coordinate within the CCD segment
     * @param parallel The parallel coordinate within the CCD segment
     */
    void setSegmentSerialParallel(Segment segment, int serial, int parallel);
    /**
     * The CCD segment corresponding to the currently selected coordinate.
     * @return The segment
     */
    Segment getSegment();
    /**
     * The serial position of the currently selected pixel within the current CCD segment.
     * @return The serial coordinate
     */
    int getSerial();
    /**
     * The parallel coordinate of the currently selected pixel within the current CCD segment.
     * @return The parallel coordinate
     */
    int getParallel();
    /**
     * The type of pixel corresponding to the currently selected coordinate.
     * @return The pixel type
     */
    PixelType getPixelType();
    /**
     * The global coordinate corresponding to the currently selected coordinate. 
     * If the CCDTransform was obtained via {@link CCDGeometry#getGlobalTransform()} then 
     * it will be equivalent the the x coordinate, but if it was obtained via 
     * {@link CCDGeometry#getActiveParallelSize()} it will return the global x coordinate
     * corresponding to the currently selected active pixel.
     * @return The global x coordinate
     */
    int getGlobalX();
    /**
     * The global coordinate corresponding to the currently selected coordinate. 
     * If the CCDTransform was obtained via {@link CCDGeometry#getGlobalTransform()} then 
     * it will be equivalent the the y coordinate, but if it was obtained via 
     * {@link CCDGeometry#getActiveParallelSize()} it will return the global y coordinate
     * corresponding to the currently selected active pixel.
     * @return The global y coordinate
     */ 
    int getGlobalY();
    
    /**
     * Get the ReadOutParameters object used internally by the CCDTransform instance.
     * 
     * @return the ReadOutParameters object.
     */
    ReadOutParameters getReadOutParameters();
    
    /**
     * Create a CCDTransform which considers all pixels for the absolute x,y
     * coordinates.
     *
     * @param ccd The CCD on which this transformation acts.
     * @param readOutParameters The ReadOutParameters used to define the pre and over scan regions
     * @return The CCDTransform for global coordinates
     */
    public static CCDTransform getGlobalTransform(CCD ccd, ReadOutParameters readOutParameters) {
        return new GlobalTransform(ccd, readOutParameters);
    } 

    /**
     * Create a CCDTransform which considers all pixels for the absolute x,y
     * coordinates.
     * Internally it will use the default ReadOutParameters for the ccd type.
     *
     * @param ccd The CCD on which this transformation acts.
     * @return The CCDTransform for global coordinates
     */
    public static CCDTransform getGlobalTransform(CCD ccd) {
        return new GlobalTransform(ccd, new ReadOutParameters(ccd.getType()));
    } 

    /**
     * Create a CCDTransform which only considers active pixels for the absolute
     * x,y coordinates. When using this transform PixelType will always be
     * active. Setting a CCD coordinate which does not correspond to an active
     * pixel will result in an error. The {@link CCDTransform#getGlobalX()} and
     * {@link CCDTransform#getGlobalY()} methods can be used to obtain the
     * global coordinate corresponding to the x,y given.
     *
     * @param ccd The CCD on which this transformation acts.
     * @param readOutParameters The ReadOutParameters used to define the pre and over scan regions
     * @return The CCDTransform for active coordinates.
     */
    public static CCDTransform getActiveTransform(CCD ccd, ReadOutParameters readOutParameters) {
        return new ActiveTransform(ccd, readOutParameters);
    }

    /**
     * Create a CCDTransform which only considers active pixels for the absolute
     * x,y coordinates. When using this transform PixelType will always be
     * active. Setting a CCD coordinate which does not correspond to an active
     * pixel will result in an error. The {@link CCDTransform#getGlobalX()} and
     * {@link CCDTransform#getGlobalY()} methods can be used to obtain the
     * global coordinate corresponding to the x,y given.
     * 
     * Internally it will use the default ReadOutParameters for the ccd type.
     *
     * @param ccd The CCD on which this transformation acts.
     * @return The CCDTransform for active coordinates.
     */
    public static CCDTransform getActiveTransform(CCD ccd) {
        return new ActiveTransform(ccd, new ReadOutParameters(ccd.getType()));
    }
    
    /**
     * A CCDTransform which only considers active pixels for the absolute x,y
     * coordinates. When using this transform PixelType will always be active.
     * Setting a CCD coordinate which does not correspond to an active pixel
     * will result in an error. The {@link #getGlobalX()} and
     * {@link #getGlobalY()} methods can be used to obtain the global coordinate
     * corresponding to the x,y given.
     */
    class ActiveTransform implements CCDTransform {

        private Segment segment;
        private int serial;
        private int parallel;
        private int x;
        private int y;
        private int globalX;
        private int globalY;
        private final CCD ccd;
        private final ReadOutParameters readOutParameters;

        ActiveTransform(CCD ccd, ReadOutParameters readOutParameters) {
            this.ccd = ccd;
            this.readOutParameters = readOutParameters;
        }
        

        @Override
        public void setXY(int x, int y) {
            assert (x >= 0 && x < ccd.getActiveSerialSize());
            assert (y >= 0 && y < ccd.getActiveParallelSize());
            this.x = x;
            this.y = y;
            int column = x / ccd.getType().getCCDGeometryConstants().getSegmentSerialActiveSize();
            int row = y / ccd.getType().getCCDGeometryConstants().getSegmentParallelActiveSize();
            segment = ccd.getChild(row,column);
            globalX = x + segment.getSerialPosition() * (readOutParameters.getSerialPrescan() + readOutParameters.getOverCols()) + (!segment.isReadoutLeft() ? readOutParameters.getSerialPrescan() : readOutParameters.getOverCols());
            globalY = y + segment.getParallelPosition() * readOutParameters.getOverRows() + (segment.isReadoutDown() ? 0 : readOutParameters.getOverRows());
            x %= segment.getSegmentSerialActiveSize();
            y %= segment.getSegmentParallelActiveSize();
            serial = readOutParameters.getSerialPrescan() + (!segment.isReadoutLeft() ? x : segment.getSegmentSerialActiveSize() - 1 - x);
            parallel = segment.isReadoutDown() ? y : segment.getSegmentParallelActiveSize() - 1 - y;
        }

        @Override
        public Segment getSegment() {
            return segment;
        }

        @Override
        public int getSerial() {
            return serial;
        }

        @Override
        public int getParallel() {
            return parallel;
        }

        @Override
        public PixelType getPixelType() {
            return PixelType.ACTIVE;
        }

        @Override
        public int getGlobalX() {
            return globalX;
        }

        @Override
        public int getGlobalY() {
            return globalY;
        }

        @Override
        public String toString() {
            return "ActiveTransform{" + "segment=" + segment + ", serial=" + serial + ", parallel=" + parallel + ", x=" + x + ", y=" + y + '}';
        }

        @Override
        public void setSegmentSerialParallel(Segment segment, int serial, int parallel) {
            assert (segment.getParent() == ccd);
            assert (0 <= parallel && parallel < segment.getSegmentParallelActiveSize());
            assert (readOutParameters.getSerialPrescan() <= serial && serial < readOutParameters.getSerialPrescan() + segment.getSegmentSerialActiveSize());
            this.segment = segment;
            this.serial = serial;
            this.parallel = parallel;
            serial -= readOutParameters.getSerialPrescan();
            int yy = segment.isReadoutDown() ? parallel : segment.getSegmentParallelActiveSize() - 1 - parallel;
            int xx = !segment.isReadoutLeft() ? serial : segment.getSegmentSerialActiveSize() - 1 - serial;
            yy += segment.getParallelPosition()* segment.getSegmentParallelActiveSize();
            xx += segment.getSerialPosition()* segment.getSegmentSerialActiveSize();
            this.x = xx;
            this.y = yy;
            globalX = x + segment.getSerialPosition() * (readOutParameters.getSerialPrescan() + readOutParameters.getOverCols()) + (!segment.isReadoutLeft() ? readOutParameters.getSerialPrescan() : readOutParameters.getOverCols());
            globalY = y + segment.getParallelPosition() * readOutParameters.getOverRows() + (segment.isReadoutDown() ? 0 : readOutParameters.getOverRows());
        }

        @Override
        public int getX() {
            return x;
        }

        @Override
        public int getY() {
            return y;
        }

        @Override
        public ReadOutParameters getReadOutParameters() {
            return readOutParameters;
        }
        
        
    }

    /**
     * A CCDTransform which considers all pixels for the absolute x,y
     * coordinates.
     */
    class GlobalTransform implements CCDTransform {

        private Segment segment;
        private int serial;
        private int parallel;
        private PixelType pixelType;
        private int x;
        private int y;
        private final CCD ccd;
        private final ReadOutParameters readOutParameters;
        
        GlobalTransform(CCD ccd, ReadOutParameters readOutParameters) {
            this.ccd = ccd;
            this.readOutParameters = readOutParameters;
        }

        @Override
        public void setXY(int x, int y) {
            assert (x >= 0 && x < ReadOutParameters.getCCDTotalSerialSize(ccd,readOutParameters));
            assert (y >= 0 && y < ReadOutParameters.getCCDTotalParallelSize(ccd,readOutParameters));
            this.x = x;
            this.y = y;
            int column = x / readOutParameters.getTotalSerialSize();
            int row = y / readOutParameters.getTotalParallelSize();
            segment = ccd.getChild(row,column);
            y %= readOutParameters.getTotalParallelSize();
            x %= readOutParameters.getTotalSerialSize();
            parallel = segment.isReadoutDown() ? y : readOutParameters.getTotalParallelSize() - 1 - y;
            serial = segment.isReadoutLeft() ? x : readOutParameters.getTotalSerialSize() - 1 - x;
            if (parallel > segment.getSegmentParallelActiveSize()) {
                pixelType = PixelType.PARALLEL_OVERSCAN;
            } else if (serial < readOutParameters.getSerialPrescan()) {
                pixelType = PixelType.SERIAL_PRESCAN;
            } else if (serial > readOutParameters.getSerialPrescan() + segment.getSegmentSerialActiveSize()) {
                pixelType = PixelType.SERIAL_OVERSCAN;
            } else {
                pixelType = PixelType.ACTIVE;
            }
        }

        @Override
        public void setSegmentSerialParallel(Segment segment, int serial, int parallel) {
            assert (segment.getParent() == ccd);
            assert (0 <= parallel && parallel < readOutParameters.getTotalParallelSize());
            assert (0 <= serial && serial < readOutParameters.getTotalSerialSize());
            this.segment = segment;
            this.serial = serial;
            this.parallel = parallel;
            if (parallel > segment.getSegmentParallelActiveSize()) {
                pixelType = PixelType.PARALLEL_OVERSCAN;
            } else if (serial < readOutParameters.getSerialPrescan()) {
                pixelType = PixelType.SERIAL_PRESCAN;
            } else if (serial > readOutParameters.getSerialPrescan() + segment.getSegmentSerialActiveSize()) {
                pixelType = PixelType.SERIAL_OVERSCAN;
            } else {
                pixelType = PixelType.ACTIVE;
            }
            int xx = segment.isReadoutLeft() ? serial : readOutParameters.getTotalSerialSize() - 1 - serial;
            int yy = segment.isReadoutDown() ? parallel : readOutParameters.getTotalParallelSize() - 1 - parallel;
            yy += segment.getParallelPosition()* readOutParameters.getTotalParallelSize();
            xx += segment.getSerialPosition()* readOutParameters.getTotalSerialSize();
            this.x = xx;
            this.y = yy;
        }

        @Override
        public Segment getSegment() {
            return segment;
        }

        @Override
        public int getSerial() {
            return serial;
        }

        @Override
        public int getParallel() {
            return parallel;
        }

        @Override
        public PixelType getPixelType() {
            return pixelType;
        }

        @Override
        public String toString() {
            return "GlobalTransform{" + "segment=" + segment + ", serial=" + serial + ", parallel=" + parallel + ", pixelType=" + pixelType + ", x=" + x + ", y=" + y + '}';
        }

        @Override
        public int getGlobalX() {
            return x;
        }

        @Override
        public int getGlobalY() {
            return y;
        }

        @Override
        public int getX() {
            return x;
        }

        @Override
        public int getY() {
            return y;
        }

        @Override
        public ReadOutParameters getReadOutParameters() {
            return readOutParameters;
        }
    }
}
