package org.lsst.ccs.utilities.ccd;

import java.awt.Dimension;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * A class describing a CCD.
 * Its default constructor is designed to contain 2 Segments along the parallel
 * position and 8 in the serial direction.
 * 
 * The CCD dimensions and the Segments readout parameters are described in the
 * CCDType provided at construction time.
 * 
 * @author The LSST CCS Team.
 */
public class CCD extends Geometry<Segment> {

    private final CCDType ccdType;
    private final List<Segment> segments = new ArrayList<>();

    /**
     * Create an empty CCD object. It takes the CCDType describing
     * all the geometry parameter for this CCD and the readout parameters for
     * the embedded Segments.
     * 
     * @param ccdType The CCDType for this CCD.
     */
    CCD(CCDType ccdType) {
        this(CCDGeometryConstants.SEGMENTS_ALONG_PARALLEL, CCDGeometryConstants.SEGMENTS_ALONG_SERIAL, ccdType);
    }

    /**
     * Create an empty CCD object. To create geometry objects for standard CCD
     * types see the {@link CCDType} class.
     *
     * @param serialSegmentsCount The number of segments in the serial direction
     * @param parallelSegmentsCount The number of segments in the parallel direction
     * @param ccdType The type of this CCD.
     */
    CCD(int parallelSegmentsCount, int serialSegmentsCount, CCDType ccdType) {
        super("S", new Dimension(ccdType.getCCDGeometryConstants().getPhysicalAreaParallelSize(), ccdType.getCCDGeometryConstants().getPhysicalAreaSerialSize()),parallelSegmentsCount, serialSegmentsCount);
        this.ccdType = ccdType;
    }

    /**
     * Add a Segment to this CCD at a given serial and parallel position.
     *
     * @param segment The Segment to add to the CCD.
     * @param s The Segment serial position.
     * @param p The Segment parallel position.
     */
    @Override
    protected void addGeometryToGrid(Segment segment, int p, int s) {
        int x = (int)ccdType.getCCDGeometryConstants().getParallelEdge() + (getParallelChildrenCount()-1-p)*segment.getWidth();
        int y = (int)ccdType.getCCDGeometryConstants().getSerialEdge()+(getSerialChildrenCount()-1-s) * segment.getHeight();
        addGeometry(segment, x, y);
        segments.add(segment);
        Collections.sort(segments, (Segment o1, Segment o2) -> o1.getChannel() > o2.getChannel() ? 1 : -1);

    }

    /**
     * The active serial pixel size of this CCD for all the Segments.
     * It is the geometrical pixel size for all Segments.
     * It does not include prescan and overscan in the serial direction.
     * 
     * It does not count for gaps at the edges.
     * 
     * @return 
     */
    public int getActiveSerialSize() {
        return ccdType.getCCDGeometryConstants().getSegmentSerialActiveSize() * getSerialChildrenCount();
    }

    /**
     * The active parallel pixel size of this CCD for all the Segments.
     * It is the geometrical pixel size for all Segments.
     * It does not the overscan in the parallel direction.
     * 
     * It does not count for gaps at the edges.
     * 
     * @return 
     */
    public int getActiveParallelSize() {
        return ccdType.getCCDGeometryConstants().getSegmentParallelActiveSize() * getParallelChildrenCount();
    }

    /**
     * Get the CCDType for this CCD.
     * 
     * @return The CCDType for this CCD.
     */
    public CCDType getType() {
        return ccdType;
    }

    /**
     * Get the ordered list for all the Segments in this CCD. The list order
     * is ascending using the Segment channel information. It is the order
     * in which they are written.
     * 
     * @return The order list of the Segments in the CCD.
     */
    public List<Segment> getSegments() {
        return Collections.unmodifiableList(segments);
    }

    /**
     * Utility method to create a CCD for a given CCDType filled with Segments.
     * 
     * @param ccdType The CCDType for this CCD
     * @return        A fully Segment populated CCD for the given CCDType.
     */
    public static CCD createCCD(CCDType ccdType) {

        CCD ccd = new CCD(ccdType);
        for (int s = 0; s < ccd.getSerialChildrenCount(); s++) {
            for (int p = 0; p < ccd.getParallelChildrenCount(); p++) {
                int channel = p == 0 ? 16 - s : 1 + s;
                Segment segment = Segment.createCCDSegment(ccdType, evaluateSegmentReadOutOrder(p, ccdType), channel);
                ccd.addChildGeometry(segment, p, s);
            }
        }
        return ccd;
    }
    
    /**
     * Evaluate a Segment readout order which depends on the type of CCD we
     * are using and on the parallel position of the Segment itself.
     * 
     * @param parallelPosition The Segment parallel position.
     * @param type             The CCD type
     * @return The corresponding SegmentReadOutOrder
     * 
     */
    private static SegmentReadOutOrder evaluateSegmentReadOutOrder(int parallelPosition, CCDType type) {
        if (parallelPosition == 1) {
            return SegmentReadOutOrder.LeftUp;
        } else {
            return type.getName().toLowerCase().contains("e2v") ? SegmentReadOutOrder.RightDown : SegmentReadOutOrder.RightUp;
        }
        
    }

}
