package org.lsst.ccs.utilities.image;

import org.lsst.ccs.utilities.ccd.CCD;
import org.lsst.ccs.utilities.ccd.CCDType;
import org.lsst.ccs.utilities.ccd.E2VCCDType;
import org.lsst.ccs.utilities.ccd.ITLCCDType;
import org.lsst.ccs.utilities.logging.Logger;


/**
 * An object that defines the read out parameters used to acquire an ImageSet.
 * These are the parameters that are set on the Sequencer at the time
 * of the acquisition.
 * It also needs the CCDType to evaluate other quantities to be used when
 * writing out fits files.
 * 
 * These parameters are at the Segment level.
 * 
 * @author The LSST CCS Team
 */
public final class ReadOutParameters {
    
    //Parallel direction readout parameters
    private int preRows, readRows, postRows;
    //Serial direction readout parameters
    private int preCols, readCols, postCols, readCols2;
    //Parallel and serial over scan
    private int overRows, overCols;
    //The CCD type for this ReadOutParameters object
    private CCDType ccdType;
    
    private static final Logger log = Logger.getLogger("org.lsst.ccs.utilities.image.readoutparameters");

    /**
     * Segment redout parameters quantities according to 
     * https://jira.slac.stanford.edu/projects/LSSTCCSRAFTS/issues/LSSTCCSRAFTS-67.
     * 
     * The following parameters describe how each segment was read.
     * TO-DO Add reference to diagram from confluence.
     * TO-DO Add validate method to make sure the parameters meet the constraints
     * listed at: 	https://confluence.slac.stanford.edu/x/hrjdD
     * 
     * They are used both for full frame readout (the whole segment is read) and
     * for ROI. readCols2 is used when an ROI is across segments.
     * 
     * @param preRows  Rows skipped 
     * @param readRows Rows read
     * @param postRows Rows discarded after the read rows
     * @param overRows Rows read in over scan
     * @param preCols  Columns skipped (this includes the pre scan columns)
     * @param readCols Columns read
     * @param postCols Columns discarded after the read columns
     * @param readCols2 Columns read for the second sub-array
     * @param overCols Columns read in over scan
     * @param ccdType The CCD Type
     */
    public ReadOutParameters(CCDType ccdType, int preRows, int readRows, int postRows, int overRows,
            int preCols, int readCols, int postCols, int readCols2, int overCols) {
        this.preRows = preRows;
        this.readRows = readRows;
        this.postRows = postRows;
        this.overRows = overRows;
        this.preCols = preCols;
        this.readCols = readCols;
        this.postCols = postCols;
        this.overCols = overCols;
        this.readCols2 = readCols2;
        this.ccdType = ccdType;
        validate();
    }    

    
    /**
     * Create a full frame ReadOutParameters object for a given type of CCD
     * 
     * @param type The CCD type.
     */
    public ReadOutParameters(CCDType type) {
        this(type, 0,type.getCCDGeometryConstants().getSegmentParallelActiveSize(),0, getParallelOverscan(type),
                0, type.getCCDGeometryConstants().getSegmentSerialPrescanSize()+type.getCCDGeometryConstants().getSegmentSerialActiveSize(), 0, 0, getSerialOverscan(type));  
        validate();
    }
    
    /**
     * Get the number of the skipped rows.
     * 
     * @return The number of skipped rows.
     */
    public int getPreRows() {
        return preRows;
    }

    /**
     * Set the number of the skipped rows.
     * @param preRows The number of skipped rows.
     */
    public void setPreRows(int preRows) {
        this.preRows = preRows;
    }

    /**
     * Get the number of the read rows.
     * 
     * @return The number of read rows.
     */
    public int getReadRows() {
        return readRows;
    }

    /**
     * Set the number of the read rows.
     * 
     * @param readRows The number of read rows.
     */
    public void setReadRows(int readRows) {
        this.readRows = readRows;
    }

    /**
     * Get the number of the discarded rows after the ones that have been read.
     * 
     * @return The number of discarded rows after the ones that have been read.
     */
    public int getPostRows() {
        return postRows;
    }

    /**
     * Set the number of the discarded rows after the ones that have been read.
     * 
     * @param postRows The number of discarded rows after the ones that have been read.
     */
    public void setPostRows(int postRows) {
        this.postRows = postRows;
    }

    /**
     * Get the number of the skipped columns.
     * 
     * @return The number of skipped columns.
     */
    public int getPreCols() {
        return preCols;
    }

    /**
     * Set the number of the skipped columns.
     * @param preCols The number of skipped columns.
     */
    public void setPreCols(int preCols) {
        this.preCols = preCols;
    }

    /**
     * Get the number of the read columns.
     * 
     * @return The number of read columns.
     */
    public int getReadCols() {
        return readCols;
    }

    /**
     * Set the number of the read columns.
     * 
     * @param readCols The number of read columns.
     */
    public void setReadCols(int readCols) {
        this.readCols = readCols;
    }

    /**
     * Get the number of the discarded columns after the ones that have been read.
     * 
     * @return The number of discarded columns after the ones that have been read.
     */
    public int getPostCols() {
        return postCols;
    }

    /**
     * Set the number of the discarded columns after the ones that have been read.
     * 
     * @param postCols The number of discarded columns after the ones that have been read.
     */
    public void setPostCols(int postCols) {
        this.postCols = postCols;
    }

    /**
     * Get the number of the read columns.
     * 
     * @return The number of read columns.
     */
    public int getReadCols2() {
        return readCols2;
    }

    /**
     * Set the number of the read columns for the second sub-array.
     * 
     * @param readCols2 The number of the read columns for the second sub-array.
     */
    public void setReadCols2(int readCols2) {
        this.readCols2 = readCols2;
    }

    /**
     * Get the number of the read columns for the second sub-array.
     * This is the parallel over scan size.
     * 
     * @return The number of the read columns for the second sub-array.
     */
    public int getOverRows() {
        return overRows;
    }

    /**
     * Set the number of the read rows in over scan.
     * 
     * @param overRows The number of read rows in over scan.
     */
    public void setOverRows(int overRows) {
        this.overRows = overRows;
    }

    /**
     * Get the number of the read columns in over scan.
     * These is the serial over scan size.
     * 
     * @return The number of read columns in over scan.
     */
    public int getOverCols() {
        return overCols;
    }

    /**
     * Set the number of the read columns in over scan.
     * 
     * @param overCols The number of read columns in over scan.
     */
    public void setOverCols(int overCols) {
        this.overCols = overCols;
    }
    
    /**
     * UTILITY METHODS USED WHE WRITING FITS FILES.
     */

    /**
     * Utility method to get the the number of pixels read in the serial
     * direction. This is the sum of the read and over scan pixels.
     * Notice that readCols includes pre scan pixels in full frame mode.
     * 
     * @return the number of pixels read in the serial direction.
     */
    public int getSerialReadPixels() {
        return getReadCols() + getReadCols2() + getOverCols();
    }
    
    /**
     * Utility method to get the the number of pixels read in the parallel
     * direction. This is the sum of the read and over scan pixels.
     * 
     * @return the number of pixels read in the parallel direction.
     */
    public int getParallelReadPixels() {
        return getReadRows() + getOverRows();
    }
    
    /**
     * Utility method to get the serial register size.
     * It's the sum of preCols, readCols, postCols and readCols2.
     * 
     * @return the serialRegister size
     */
    public int getSerialRegister() {
        return getPreCols()+getReadCols()+getPostCols()+getReadCols2();
    }
    
    /**
     * Get the serial prescan size.
     * This corresponds to Designator variable "preh" in Table 8 of LCA-13501
     * 
     * The difference between the serial prescan size and the preCols quantity is
     * that the serial prescan is read in prescan mode, while the preCols are
     * skipped.
     * @return The serial prescan size.
     */
    public int getSerialPrescan() {
        return ccdType.getCCDGeometryConstants().getSegmentSerialPrescanSize();
    }

    /**
     * Get the total serial size in pixels.
     * This is the sum of the serial pre scan, active and over scan sizes.
     * 
     * @return The total serial size in pixels.
     */
    public int getTotalSerialSize() {
        return getSerialRegister()+getOverCols();
    }

    /**
     * Get the total parallel size in pixels.
     * This is the sum of the parallel over scan and the active sizes.
     * 
     * @return The total serial size in pixels.
     */
    public int getTotalParallelSize() {
        return ccdType.getCCDGeometryConstants().getSegmentParallelActiveSize()+getOverRows();
    }

    /**
     * Utility method to get the parallel over scan for a given CCD type.
     * @param type The CCD type.
     * @return     The parallel over scan
     */    
    private static int getParallelOverscan(CCDType type) {
        if ( type instanceof E2VCCDType ) {
            return 46;
        } else if ( type instanceof ITLCCDType ) {
            return 48;
        } else {
            return 0;
        }
    }
            
    /**
     * Utility method to get the serial over scan for a given CCD type.
     * @param type The CCD type.
     * @return     The serial over scan
     */    
    private static int getSerialOverscan(CCDType type) {
        if ( type instanceof E2VCCDType ) {
            return 22;
        } else if ( type instanceof ITLCCDType ) {
            return 32;
        } else {
            return 0;
        }
    }
    
    /**
     * Utility method to get a CCD full serial size for a given ReadOutParameter.
     * The full serial size of a CCD is the product of a segment total serial
     * size and the number of segments in the serial direction.
     * 
     * This size includes active, pre and post scan sizes.
     * 
     * @param ccd The CCD instance.
     * @param readOutParameters The ReadOutParameters.
     * @return    The CCD full serial size including pre and over scans.
     */    
    public static int getCCDTotalSerialSize(CCD ccd, ReadOutParameters readOutParameters) {
        return readOutParameters.getTotalSerialSize() * ccd.getSerialChildrenCount();
    }

    /**
     * Utility method to get a CCD full parallel size for a given ReadOutParameter.
     * The full parallel size of a CCD is the product of a segment total parallel
     * size and the number of segments in the parallel direction.
     * 
     * This size includes active and over scan sizes.
     * 
     * @param ccd The CCD instance.
     * @param readOutParameters The ReadOutParameters.
     * @return    The CCD full parallel size including over scan.
     */    
    public static int getCCDTotalParallelSize(CCD ccd, ReadOutParameters readOutParameters) {
        return readOutParameters.getTotalParallelSize() * ccd.getParallelChildrenCount();
    }
    
    /**
     * Set the CCDType for this ReadOutParameter object.
     * @param ccdType The CCD type.
     */
    public void setCCDType(CCDType ccdType) {
        this.ccdType = ccdType;
    }
    
    /**
     * Validate the current set of readout parameters.
     * For now we print a WARNING if the parameters are not valid.
     * 
     */
    private void validate() {
        boolean isValid = true;
        int totalParallelPixels = ccdType.getCCDGeometryConstants().getSegmentParallelActiveSize();
        if ( getPreRows() + getReadRows() + getPostRows() != totalParallelPixels ) {
            log.warn("Parallel parameters are invalid: preRows("+getPreRows()
                    + ") + readRows("+getReadRows()+") + postRows("+getPostRows()+") != segRows("+totalParallelPixels+")");
            isValid = false;
        }
        int totalSerialPixels = ccdType.getCCDGeometryConstants().getSegmentSerialActiveSize()+ccdType.getCCDGeometryConstants().getSegmentSerialPrescanSize();
        if ( getPreCols() + getReadCols() + getPostCols() + getReadCols2() !=  totalSerialPixels ) {
            log.warn("Serial parameters are invalid: preCols("+getPreCols()
                    + ") + readCols("+getReadCols()+") + readCols2("+getReadCols2()+") + postCols("+getPostCols()+") != "
                    + "serCols("+totalSerialPixels+")");
            isValid = false;
        }
        if ( !isValid ) {
            log.warn("The geometry values in the Fits Headers will be incorrect.");
        }
                
    }
}
