package org.lsst.ccs.utilities.image;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.FutureTask;
import nom.tam.fits.BasicHDU;
import nom.tam.fits.FitsException;
import nom.tam.fits.FitsFactory;
import nom.tam.fits.FitsUtil;
import nom.tam.fits.Header;
import nom.tam.fits.HeaderCardException;
import nom.tam.util.BufferedFile;
import org.lsst.ccs.utilities.ccd.image.data.RawImageData.BitsPerPixel;
import static org.lsst.ccs.utilities.image.HeaderSpecification.DataType.Float;
import org.lsst.ccs.utilities.image.HeaderSpecification.HeaderLine;

/**
 * A utility for writing FITS files following LSST conventions.
 *
 * @author tonyj
 */
public class FitsFileWriter implements Closeable {

    private class ImageExtension {

        private final long startPosition;
        private final long endPosition;
        private long currentPosition;
        private final FitsCheckSum.Checksum checkSum;
        private final Header header;
        private final BufferedFile file;

        private ImageExtension(BufferedFile file, long start, long current, Header header) {
            this.startPosition = start;
            this.currentPosition = current;
            this.endPosition = file.getFilePointer();
            this.header = header;
            this.checkSum = new FitsCheckSum.Checksum();
            this.file = file;
        }

        //TO-DO: to allow parallel writing we need a method that takes the offset
        //position where to write and the bytebuffer containing the data to write
        //The checksum is an extension feature. The checksum depends on the order in
        //which it is filled. So that would be a problem with random access 
        //writing. So maybe we should skip calculating the checksum.
        
        private FutureTask<Void> write(ByteBuffer src) throws IOException {
            FutureTask<Void> future = new FutureTask(() -> {
                try {
                    int length = src.remaining();
                    if (length + currentPosition > endPosition) {
                        throw new IOException("Too much data written for image " + length + " " + currentPosition + " " + endPosition);
                    }
                    FitsCheckSum.updateChecksum(src, checkSum);
                    file.seek(currentPosition);
                    if (src.hasArray()) {
                        file.write(src.array(), src.arrayOffset() + src.position(), src.remaining());
                        src.position(src.limit());
                    } else {
                        while (src.remaining() > 0) {
                            file.write(src.get());
                        }
                    }
                    currentPosition += length;
                } catch (IOException ioe) {
                    throw new RuntimeException(ioe);
                }
            },true);
            future.run();
            return future;
        }

        //Invoked before closing the file.
        private void updateDataSum() throws IOException {
            try {
                FitsCheckSum.updateDataSum(header, checkSum.getCheckSum());
                file.seek(startPosition);
                header.write(file);
            } catch (FitsException ex) {
                throw new IOException("Unable to add datasum to header", ex);
            }
        }
    }
    private BufferedFile bf;
    private ImageExtension[] imageExtensions;

    private void initializeFitsFileWriter(File file, ImageSet images, MetaDataSet metaDataSet, Map<String, HeaderSpecification> config, List<FitsHeaderMetadataProvider> providers, BitsPerPixel bits) throws IOException, FitsException {

        List<FitsHeaderMetadataProvider> inner_providers = new ArrayList<>();
        FitsHeaderMetadataProvider geometryProvider = new GeometryFitsHeaderMetadataProvider();
        inner_providers.add(geometryProvider);
        if (providers != null) {
            inner_providers.addAll(providers);
        }
        
        FitsHeaderMetadataProvider inner_provider = new CompositeFitsHeaderMetadataProvider(inner_providers);
        
        
        imageExtensions = new ImageExtension[images.getNumberOfImages()];
        int[][] intDummyData = new int[1][1];
        short[][] shortDummyData = new short[1][1];
        Object[] tableDummyData = new Object[0];
        Object dummyData = bits == BitsPerPixel.BIT16 ? shortDummyData : intDummyData;
        bf = new BufferedFile(file, "rw");

        FitsFactory.setUseHierarch(true);

        MetaDataSet fullMetaData = inner_provider.getPrimaryHeaderMetadata(images);
        if (metaDataSet != null) {
            fullMetaData.addMetaDataSet(metaDataSet);
        }

        // Create primary header
        BasicHDU primary = BasicHDU.getDummyHDU();
        addMetaDataToHeader(primary, "primary", fullMetaData, config);
        FitsCheckSum.setChecksum(primary);
        primary.getHeader().write(bf);

        int serialPixels = images.getReadOutParameters().getSerialReadPixels();
        int parallelPixels = images.getReadOutParameters().getParallelReadPixels();
        
        //Create extension image headers for each image, and reserve space for data
        for ( int i = 0; i < images.getNumberOfImages(); i++ ) {
            BasicHDU hdu = FitsFactory.HDUFactory(dummyData);
            
            MetaDataSet extendedMetadata = inner_provider.getDataExtendedHeaderMetadata(images,i);
            if (metaDataSet != null) {
                extendedMetadata.addMetaDataSet(metaDataSet);
            }
            addMetaDataToHeader(hdu, "extended", extendedMetadata, config);
            if (bits == BitsPerPixel.BIT16) {
                // To store as unsigned 16 bit values, we have to set BZERO to 32768, and 
                // subtract 32768 from each value. 
                // See: http://heasarc.gsfc.nasa.gov/docs/software/fitsio/c/c_user/node23.html
                hdu.addValue("BSCALE", 1.0, "Unsigned 16 bit data");
                hdu.addValue("BZERO", 32768, "Unsigned 16 bit data");
            } else {
                hdu.addValue("BSCALE", 1.0, "");
                hdu.addValue("BZERO", 0.0, "");                
            }
            Header header = hdu.getHeader();
            header.setXtension("IMAGE");
            FitsCheckSum.setChecksum(hdu);
            long start = bf.getFilePointer();
            header.write(bf);
            long current = bf.getFilePointer();
            long imageSize = bits.bytes() * serialPixels * parallelPixels;
            bf.seek(bf.getFilePointer() + imageSize);
            FitsUtil.pad(bf, imageSize);
            BufferedFile imageExtensionBufferedFile = new BufferedFile(file, "rw");
            imageExtensionBufferedFile.seek(bf.getFilePointer());
            imageExtensions[i] = new ImageExtension(imageExtensionBufferedFile, start, current, header);
        }

        // If necessary, create any additional extended HDU's here.
        // Create any extra BinTables from the specification
        FitsFactory.setUseAsciiTables(false);
        for (String key : config.keySet()) {
            if (!"primary".equals(key) && !"extended".equals(key)) {
                BasicHDU binary = FitsFactory.HDUFactory(tableDummyData);

                MetaDataSet additionalMetaData = inner_provider.getAdditionalExtendedHeaderMetadata(images,key);
                if (metaDataSet != null) {
                    additionalMetaData.addMetaDataSet(metaDataSet);
                }
                addMetaDataToHeader(binary, key, additionalMetaData, config);
                Header header = binary.getHeader();
                header.setXtension("BINTABLE");
                FitsCheckSum.setChecksum(binary);
                header.write(bf);
            }
        }
        inner_provider.completedHeaderMetadata(images);
    }

    /**
     * Construct a FitsFileWriter object for the provided ImageSet.
     * By default this will write 32 bit images.
     *
     * @param file The name of the output fits file
     * @param images The ImageSet of images to write in the Data extended headers.
     *               Note that this specifies the images to write, not the 
     *               actual data for the images.
     * @param images The ImageSet to write to the file. 
     * @param metaData The meta-data maps to use to extract header info from
     * @param bits The number of bits per pixel for images
     * @throws IOException
     * @throws FitsException
     */
    @Deprecated
    public FitsFileWriter(File file, ImageSet images) throws IOException, FitsException {
        this(file, images, null, null, BitsPerPixel.BIT32);
    }
    public FitsFileWriter(File file, ImageSet images, Map<String, HeaderSpecification> specs) throws IOException, FitsException {
        this(file, images, null, null, BitsPerPixel.BIT32, specs);
    }

    @Deprecated
    public FitsFileWriter(File file, ImageSet images, MetaDataSet metaDataSet) throws IOException, FitsException {
        this(file, images, metaDataSet, null, BitsPerPixel.BIT32);
    }

    public FitsFileWriter(File file, ImageSet images, Map<String, HeaderSpecification> specs, MetaDataSet metaDataSet) throws IOException, FitsException {
        this(file, images, metaDataSet, null, BitsPerPixel.BIT32, specs);
    }
    public FitsFileWriter(File file, ImageSet images, MetaDataSet metaDataSet, List<FitsHeaderMetadataProvider> providers) throws IOException, FitsException {
        this(file, images, metaDataSet, providers, BitsPerPixel.BIT32);
    }
    @Deprecated
    public FitsFileWriter(File file, ImageSet images, List<FitsHeaderMetadataProvider> providers) throws IOException, FitsException {
        this(file, images, null, providers, BitsPerPixel.BIT32);
    }
    public FitsFileWriter(File file, ImageSet images, Map<String, HeaderSpecification> specs, List<FitsHeaderMetadataProvider> providers) throws IOException, FitsException {
        this(file, images, null, providers, BitsPerPixel.BIT32, specs);
    }
    @Deprecated
    public FitsFileWriter(File file, ImageSet images, MetaDataSet metaDataSet, List<FitsHeaderMetadataProvider> providers, BitsPerPixel bits) throws IOException, FitsException {
        initializeFitsFileWriter(file, images, metaDataSet, FitsHeadersSpecifications.getHeaderSpecifications(), providers, bits);
    }

    public FitsFileWriter(File file, ImageSet images, MetaDataSet metaDataSet, List<FitsHeaderMetadataProvider> providers, BitsPerPixel bits, Map<String, HeaderSpecification> specs) throws IOException, FitsException {
        initializeFitsFileWriter(file, images, metaDataSet, specs, providers, bits);
    }

    /**
     * Write the actual image data to the file. It is not necessary that all of
     * the data for the image be available at once, this method will write
     * whatever data is currently available in the byte buffer to the specified
     * image, and will keep track of how much has been written to each image to
     * allow more data to be written later. This method assumes the data is
     * given in the order it is to be written to the file. If any data
     * reordering is needed it needs to be done before calling this method.
     *
     * @param imageIndex The image to which this data is to be written
     * @param src The image
     * @throws IOException If an IOException is generated, or if more data is
     * sent than was expected for a particular image.
     */
    public void write(int imageIndex, ByteBuffer src) throws IOException {
        FutureTask<Void> future = imageExtensions[imageIndex].write(src);
        try {
            future.get();
        } catch (InterruptedException | ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    public FutureTask<Void> asyncWrite(int imageIndex, ByteBuffer src) throws IOException {
        return imageExtensions[imageIndex].write(src);
    }

    @Override
    public void close() throws IOException {
        // Update the datasum keywords in place
        for (ImageExtension imageExtension : imageExtensions) {
            imageExtension.updateDataSum();
        }
        bf.close();
    }

    private void addMetaDataToHeader(BasicHDU hdu, String specName, MetaDataSet metaDataSet, Map<String, HeaderSpecification> config) throws HeaderCardException, IOException {
        HeaderSpecification spec = config.get(specName);
        if (spec == null) {
            throw new IOException("Missing specification for header: " + specName);
        }
        for (HeaderLine header : spec.getHeaders()) {
            Object value = header.getValue(metaDataSet);
            try {
                if (value != null) {
                    switch (header.getDataType()) {
                        case Integer:
                            hdu.addValue(header.getKeyword(), ((Number) value).intValue(), header.getComment());
                            break;
                        case Float:
                            double data = ((Number) value).doubleValue();
                            if (!Double.isFinite(data)) {
                                throw new IllegalArgumentException("Can not store non-finite floating point in FITS file");
                            }
                            hdu.addValue(header.getKeyword(), data, header.getComment());
                            break;
                        case Boolean:
                            hdu.addValue(header.getKeyword(), (Boolean) value, header.getComment());
                            break;
                        case Date:
                            hdu.addValue(header.getKeyword(), DateUtils.convertDateToString((Date) value), header.getComment());
                            break;
                        case MJD:
                            hdu.addValue(header.getKeyword(), DateUtils.convertDateToMJD((Date) value), header.getComment());
                            break;
                        default:
                            hdu.addValue(header.getKeyword(), String.valueOf(value), header.getComment());
                    }
                }
            } catch (ClassCastException x) {
                throw new IOException(String.format("Meta-data header %s with value %s(%s) cannot be converted to type %s", header.getKeyword(), value, value.getClass(), header.getDataType()));
            }
        }
    }

}
