
package org.lsst.ccs.utilities.image.patterns;

import java.awt.Transparency;
import java.awt.color.ColorSpace;
import java.awt.image.BufferedImage;
import java.awt.image.ComponentColorModel;
import java.awt.image.DataBuffer;
import java.awt.image.Raster;
import java.awt.image.WritableRaster;
import java.util.Map;

/**
 * The base class for an Image pattern generator.
 * A subclass must provide an empty constructor in order for the PatternGeneratorFactory to work.
 * This is so that we can create a new instance of a pattern generation 
 * on different threads, ensuring its thread safety.
 * Pattern generation parameters are passed as maps when the image is generated.
 * @author turri
 */
public abstract class PatternGenerator {
    
    private final String name;
    
    PatternGenerator(String name) {
        this.name = name;        
    }
        
    /**
     * The name of the pattern.
     * It will be used to select the pattern to generate.
     * 
     * @return The name of the pattern.
     */
    public String getName() {
        return name;
    }
    
    /**
     * Generate a GeneratedImage with the provided width and height.
     * The generated pattern is affected by the provided parameters.
     * Refer to the subclasses API for more information about the available
     * parameter. By convention the parameters start with the name of the
     * PatternGenerator.
     * 
     * @param width The image width
     * @param height The image 
     * @param parameters The parameters to be used to generate the image.
     * @return The generated GeneratedImage
     */
    public abstract GeneratedImage generateImage(int width, int height, Map<String,String> parameters);
    
    protected String getParameter(String parName, Map<String,String> parameters) {
        if ( parameters == null ) {
            return null;
        }
        return parameters.get(parName);
    }
    
    protected double getDoubleParameter(String parName, Map<String,String> parameters, double defaultValue) {
        String par = getParameter(parName, parameters);
        if ( par == null ) {
            return defaultValue;
        }
        return Double.parseDouble(par);        
    }

    protected int getIntParameter(String parName, Map<String,String> parameters, int defaultValue) {
        String par = getParameter(parName, parameters);
        if ( par == null ) {
            return defaultValue;
        }
        return Integer.parseInt(par);        
    }
    
    protected String getStringParameter(String parName, Map<String,String> parameters, String defaultValue) {
        String par = getParameter(parName, parameters);
        if ( par == null ) {
            return defaultValue;
        }
        return par;        
    }
    
    
            
    private static final ColorSpace cs = ColorSpace.getInstance(ColorSpace.CS_GRAY);
    private static final ComponentColorModel ccm = new ComponentColorModel(cs, false, false, 
            Transparency.OPAQUE, DataBuffer.TYPE_INT);
    
    static WritableRaster createRaster(int width, int height) {
        return ccm.createCompatibleWritableRaster(width, height);
    }
    
    static ComponentColorModel getComponentColorModel() {
        return ccm;
    }
    
    public static BufferedImage getBufferedImageFromRaster(WritableRaster r) {
        return new BufferedImage(ccm,r,false,null);
    }
}
