package org.lsst.ccs.utilities.ccd;

/**
 * An object that defines Raft specific constants used to place CCDs into 
 * a Raft instance.
 * These quantities are the gaps between CCDs and the gaps at the edges of the
 * Raft, both in the parallel and serial direction.
 * 
 * - gap_inx is the inter CCD gap along the parallel direction in pixels
 * - gap_iny is the inter CCD gap along the serial direction in pixels 
 * - gap_outx is the gap at the edge of the Raft along the parallel direction in pixels
 * - gap_outy is the gap at the edge of the Raft along the serial direction in pixels 
 * - raftx is the Raft full size in the parallel direction in pixels
 * - rafty is the Raft full size in the serial direction in pixels
 * 
 * These quantities are defined in LCA-13501.
 * 
 * @author The LSST CCS Team
 */
public final class RaftGeometryConstants {
    
    private final int gap_inx;
    private final int gap_iny;
    private final double gap_outx;
    private final double gap_outy;

    //The following quantities are use for CCD placement and are CCDType specific.
    private final int ccd_px;
    private final int ccd_py;
    
    /**
     * Create a RaftsGeometryConstants object.
     * @param gap_inx The inter CCD gap along the parallel direction in pixels
     * @param gap_iny The inter CCD gap along the serial direction in pixels
     * @param gap_outx The gap at edges of raft along the parallel direction in pixels
     * @param gap_outy The gap at edges of raft along the serial direction in pixels
     * @param ccd_px   The CCD total lenght in the parallel direction
     * @param ccd_py   The Raft total lenght in the serial direction
     */
    RaftGeometryConstants(int gap_inx, int gap_iny, double gap_outx, double gap_outy, int ccd_px, int ccd_py) {
        this.gap_inx = gap_inx;
        this.gap_iny = gap_iny;
        this.gap_outx = gap_outx;
        this.gap_outy = gap_outy;
        this.ccd_px = ccd_px;
        this.ccd_py = ccd_py;
    }
    
    /**
     * Get the inter CCD gap along the parallel direction in pixels.
     * 
     * @return the gap in pixels
     */
    public final int getGap_inx() {
        return gap_inx;
    }

    /**
     * Get the inter CCD gap along the serial direction in pixels.
     * 
     * @return the gap in pixels
     */
    public final int getGap_iny() {
        return gap_iny;
    }
    
    /**
     * Get the gap at the edge of the Raft along the parallel direction in pixels.
     * 
     * @return the gap in pixels
     */
    public final double getGap_outx() {
        return gap_outx;
    }

    /**
     * Get the gap at the edge of the Raft along the serial direction in pixels.
     * 
     * @return the gap in pixels
     */
    public final double getGap_outy() {
        return gap_outy;
    }

    /**
     * Get the Raft full size in the parallel direction in pixels.
     * 
     * @return The size in pixels.
     */    
    public final int getRaftx() {
        return 3*ccd_px + 2*getGap_inx() + (int)(2 * getGap_outx());
    }
    
    /**
     * Get the Raft full size in the serial direction in pixels.
     * 
     * @return The size in pixels.
     */    
    public final int getRafty() {
        return 3*ccd_py + 2*getGap_iny() + (int)(2 * getGap_outy());
    }

    //Used by the Reb to get the CCD parallel size.
    protected final int getPhysicalAreaParallelSize() {
        return ccd_px;
    }
    
    /**
     * Utility method to create a RaftGeometryConstants object for a given 
     * CCDType.
     * 
     * @param type The CCD type.
     * @return     The corresponding RaftGeometryConstants object.
     */
    static RaftGeometryConstants createRaftGeometryConstants(CCDType type) {
        if ( type instanceof E2VCCDType ) {
            return new RaftGeometryConstants(28,25,26.5,25,type.getCCDGeometryConstants().getPhysicalAreaParallelSize(),type.getCCDGeometryConstants().getPhysicalAreaSerialSize());
        } else if ( type instanceof ITLCCDType ) {
            return new RaftGeometryConstants(27,27,26,26,type.getCCDGeometryConstants().getPhysicalAreaParallelSize(),type.getCCDGeometryConstants().getPhysicalAreaSerialSize());
        } else {
            throw new IllegalArgumentException("Could not build RaftGeometryConstats for type "+type);
        }
    }
    
}
