package org.lsst.ccs.utilities.image;

import java.util.HashMap;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Properties;
import nom.tam.fits.FitsDate;
import nom.tam.fits.FitsException;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;

/**
 * A MetaDataSet is used to provide values for the fits header keywords.
 * Since there can be multiple sources of header keywords, this class provides
 * methods for adding key-value maps for a given source. The key corresponds
 * to the header keyword and the value represents it value. The source of origin
 * is used when resolving expressions for header keywords as described in spec
 * files as: ${metaMap.metaName}. "metaMap" corresponds to the source and "metaName"
 * to a key as coming from that specific source. The corresponding value is used
 * when writing out fits files.
 * 
 * Internally it uses a Map of Maps to store the meta-data.
 *
 * @author The LSST CCS Team.
 */
public class MetaDataSet {


    private final Map<String,Map<String,Object>> metaDataMaps = new HashMap<>();

    /**
     * Add a set of properties to a MetaDataSet. The properties will be
     * converted to the appropriate object type based on their value.
     *
     * @param name The name of the map to which the properties should be added
     * @param props The properties.
     */
    public void addProperties(String name, Properties props) {
        addMetaDataMap(name, convertToMetaData(props));
    }

    
    /**
     * Add a meta-data entry to a named meta-data object. The named object will
     * be created if it does not already exist.
     *
     * @param name The name of the map to which the new meta-data should be
     * added
     * @param key The key of the meta-data to add
     * @param value The value of the meta-data to add
     */
    public void addMetaData(String name, String key, Object value) {
        Map<String, Object> meta = findOrCreateMetaData(name);
        meta.put(key,value);
    }
    
    /**
     * Add a set of meta-data to a named meta-data object. The named object will
     * be created if it does not already exist.
     *
     * @param name The name of the map to which the new meta-data should be
     * added
     * @param data The meta-data to add
     */
    public void addMetaDataMap(String name, Map<String, Object> data) {
        if ( data == null ) {
            return;
        }
        Map<String, Object> meta = findOrCreateMetaData(name);
        meta.putAll(data);
    }

    /**
     * Add a MetaDataSet to an existing MetaDataSet.
     * A named set will be created if it does not exist. If it already exists
     * the content of the new MetaDataSet will be added to the existing one.
     * 
     * @param metaDataSet The MetaDataSet to add to the existing one.
     */
    public void addMetaDataSet(MetaDataSet metaDataSet) {
        if ( metaDataSet == null ) {
            return;
        }
        for ( Entry<String,Map<String,Object>> entry : metaDataSet.metaDataMaps.entrySet() ) {
            addMetaDataMap(entry.getKey(), entry.getValue());
        }
        
    }
    
    public Object getValue(String name) {
        for (Map<String, Object> metaData : metaDataMaps.values()) {
            Object result = metaData.get(name);
            if (result != null) {
                return result;
            }
        }
        return null;
    }

    public Object getValue(String map, String name) {
        if (map == null) {
            return getValue(name);
        } else {
            Map<String, Object> metaData = metaDataMaps.get(map);
            if (metaData == null) {
                return null;
            }
            return metaData.get(name);
        }
    }

    private Map<String, Object> findOrCreateMetaData(String name) {
        Map<String, Object> result = metaDataMaps.get(name);
        if (result == null) {
            result = new HashMap<>();
            metaDataMaps.put(name, result);
        }
        return result;
    }

    private Map<String, Object> convertToMetaData(Properties props) {
        Map<String, Object> result = new HashMap<>();
        // Note, this loops over all properties, including thos specified in the
        // default property list of the Properties object
        for (Object n : BootstrapResourceUtils.getAllKeysInProperties(props)) {            
            String name = (String)n;
            String value = props.getProperty(name);
            result.put(name, convertToMetaData(value));
        }
        return result;
    }

    private Object convertToMetaData(String value) {
        try {
            return Integer.decode(value);
        } catch (NumberFormatException x) {
            try {
                return Double.valueOf(value);
            } catch (NumberFormatException xx) {
                try {
                    return new FitsDate(value).toDate();
                } catch (FitsException xxx) {
                    if ("true".equalsIgnoreCase(value)) {
                        return Boolean.TRUE;
                    } else if ("false".equalsIgnoreCase(value)) {
                        return Boolean.FALSE;
                    } else {
                        return value;
                    }
                }
            }
        }
    }
}
