
package org.lsst.ccs.utilities.image;

/**
 * An interface to provide MetaDataSets to fill the content of Fits Headers.
 * According to the syntax used in header specification (spec) files the meta-data
 * is in the form of a MetaDataSet, that is a Map of Maps: the key names for 
 * the maps are defined as "mapName" and are used to determine at runtime the 
 * value of a fits keyword defined in the spec file as ${"mapName"."mapKey"}. 
 * The fits keyword is assigned the value corresponding to key "mapKey" in the 
 * map stored with key "mapName".
 * 
 * If "mapName" is not specified (i.e. ${"mapKey"}) then all the maps are searched
 * and the first one found containing key "mapKey" will yield the value.
 * 
 * This interface is designed to be able to provide all the meta-data needed to 
 * write a single fits file. 
 * 
 * Since a single fit file is meant to be written for
 * each CCD, the primary header meta-data should contain information for the CCD
 * itself, the Reb used to read it, the Raft in which it is embedded etc.
 * The extended data header should be written for each of the CCD segments, containing
 * information for the specific segment.
 * 
 * @author The LSST CCS Team
 */
public interface FitsHeaderMetadataProvider {
    
    /**
     * Get the primary Header MetaDataSet for a given ImageSet
     * 
     * @param imageSet the ImageSet for which the primary header meta-data is needed.
     * 
     * @return The primary header meta-data for provided ImageSet.
     */
    public default MetaDataSet getPrimaryHeaderMetadata(ImageSet imageSet) {
        return null;
    }
    
    /**
     * Get the extended header meta-data by index for a given ImageSet.
     * These are the headers containing the actual images. The index starts from 0.
     * 
     * @param extendedIndex The index of the extended header.
     * @param imageSet the ImageSet for which the extended header meta-data is needed.
     * 
     * @return The extended header meta-data by index for the provided ImageSet
     */
    public default MetaDataSet getDataExtendedHeaderMetadata(ImageSet imageSet, int extendedIndex) {
        return null;
    }
    
    /**
     * Get the metadata for an additional extended Headers by key for a given
     * ImageSet. The information
     * in these extended headers is stored as BinaryTables. These are all the
     * headers defined in the spec files with keywords other than "primary" and
     * "extended"; this keyword will be used in this method to identify which
     * metadata map to return.
     * 
     * @param imageSet the ImageSet for which the extended header meta-data is needed.
     * @param extendedKeyword The keyword identifying the extended header.
     * 
     * @return The metadata for the requested extended header
     */
    public default MetaDataSet getAdditionalExtendedHeaderMetadata(ImageSet imageSet, String extendedKeyword) {
        return null;
    }

    /**
     * Method invoked when the headers have been written for this ImageSet.
     * This is to be used for memory management, cleaning all the data that is not needed.
     * 
     * @param imageSet the ImageSet for which the headers have been completed.
     */
    public default void completedHeaderMetadata(ImageSet imageSet) {
        return;
    }
}
