package org.lsst.ccs.utilities.ccd;

import java.awt.Dimension;
import java.awt.Point;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import org.lsst.ccs.utilities.ccd.image.data.RawImageDataProvider;
import org.lsst.ccs.utilities.ccd.image.data.RawImageData;
import org.lsst.ccs.utilities.image.ImagePixelData;
import org.lsst.ccs.utilities.image.ReadOutParameters;

/**
 * An class to represent a CCD Segment.
 * Each Segment is defined by its geometry constants, the parameters for reading
 * it out (prescan and overscans) and the order in which it is read out. 
 * 
 * A Segment also has a channel number which is used to write the extended
 * headers in fits files.
 * 
 * These quantities are as specified in LCA-13501
 *
 * A Segment implements:
 * 
 * - ImageSensitiveArea so that it can be exposed to simulated images
 * - RawImageDataProvider because it is the source of image data
 * 
 * TO-DO: Can the Image related interfaces be moved elsewhere? Create a wrapper
 * around Segment?
 * 
 * @author The LSST CCS Team
 */
public class Segment extends Geometry implements RawImageDataProvider, ImageSensitiveArea {

    private final SegmentReadOutOrder readOutOrder;
    private ImagePixelData imagePixelData;
    private final int channel;

    /**
     * Create a Segment instance.
     * TO-DO Where is channel defined? Ask Seth where we can find the definition.
     * 
     * @param segmentGeometryConstants The Segment Geometry information
     * @param readOutOrder the read out order for this segment
     * @param channel The Channel number of this Segment.
     */
    Segment(SegmentGeometryConstants segmentGeometryConstants, SegmentReadOutOrder readOutOrder, int channel) {
        super("Seg", new Dimension(segmentGeometryConstants.getSegmentParallelActiveSize(),segmentGeometryConstants.getSegmentSerialActiveSize()));

        this.readOutOrder = readOutOrder;        
        this.channel = channel;
    }
    
    /**
     * The Segment channel number.
     * 
     * @return The Segment channel
     */
    public int getChannel() {
        return channel;
    }

    /**
     * The Segment active serial size in pixels.
     * 
     * @return The Segment active serial size.
     */
    public int getSegmentSerialActiveSize() {
        return getHeight();
    }

    /**
     * The Segment active parallel size in pixels.
     * 
     * @return The Segment parallel serial size.
     */
    public int getSegmentParallelActiveSize() {
        return getWidth();
    }

    
    /**
     * Returns true if the Segment is read from top to bottom in the serial
     * direction.
     * 
     * @return true if the segment is read down along the serial direction.
     */
    public boolean isReadoutDown() {
        return readOutOrder.isReadoutDown();
    }

    /**
     * Returns true if the Segment is read from left to right in the parallel
     * direction.
     * 
     * @return true if the segment is read from the left along the parallel direction.
     */
    public boolean isReadoutLeft() {
        return readOutOrder.isReadoutLeft();
    }

    //Segments don't allow children
    @Override
    protected void addGeometryToGrid(Geometry child, int p, int s) {
        throw new UnsupportedOperationException("Cannot add a geometry to a Segment.");
    }

    /**
     * Get the buffer of image data accumulated by this Segment.
     *
     * @return The ByteBuffer with the data;
     */
    @Override
    public RawImageData getRawImageData(ReadOutParameters readOutParameters) {
        DataStreamer streamer = new DataStreamer(this, readOutParameters);
        ByteBuffer dest = createByteBuffer(readOutParameters.getTotalSerialSize() * readOutParameters.getTotalParallelSize());
        for (int row = 0; row < getSegmentParallelActiveSize(); row++) {
            streamer.fillSerialPrescan(dest);
            streamer.fillDataRow(dest);
            streamer.fillSerialOverscan(dest);
        }
        streamer.fillParallelOverscan(dest);
        dest.flip();
        return new RawImageData(RawImageData.BitsPerPixel.BIT32, dest);
    }

    
    /**
     * This class has to provide the readout bytes in the order in which the
     * segments are read out. So it has to convert a rectangular image with
     * (0,0) in the top left corner to a stream corresponding to the data flow.
     * The data in the rectangle has to be ordered appropriately depending on the
     * segment's ReadOutPoint value:
     * 
     * - if the readout is to the right the image rectangle has to be read from left
     *   to right, otherwise from left to right.
     * - if the readout is down, the image rectangle has to be read top to bottom,
     *   otherwise bottom to top.
     *  
     * TO-DO review this class to include ROI properly.
     * 
     */
    private class DataStreamer {

        private final ByteBuffer serialPrescanBuffer;
        private final ByteBuffer serialOverscanBuffer;
        private final ByteBuffer parallelOverscanBuffer;
        private final ImagePixelData imagePixelData;
        private int currentImageColumn;
        private boolean increaseImageColumnCount;
        private boolean scanColumnFromTop;
        private final Segment s;
        private final ReadOutParameters readOutParameters;

        DataStreamer(Segment s, ReadOutParameters readOutParameters) {
            this.readOutParameters = readOutParameters;
            serialPrescanBuffer = createByteBuffer(readOutParameters.getSerialPrescan());
            serialOverscanBuffer = createByteBuffer(readOutParameters.getOverCols());
            int parallelOverscanBufferSize = readOutParameters.getOverRows() * (readOutParameters.getReadCols() + readOutParameters.getReadCols2() + readOutParameters.getOverCols());
            parallelOverscanBuffer = createByteBuffer(readOutParameters.getOverRows() * parallelOverscanBufferSize);

            for (int i = 0; i < readOutParameters.getSerialPrescan(); i++) {
                serialPrescanBuffer.putInt(0);
            }
            for (int i = 0; i < readOutParameters.getOverCols(); i++) {
                serialOverscanBuffer.putInt(0);
            }
            for (int i = 0; i < parallelOverscanBufferSize; i++) {
                parallelOverscanBuffer.putInt(0);
            }
            imagePixelData = s.getImagePixelData();

            this.s = s;
            reset();
        }

        public final void reset() {
            
            
            SegmentReadOutOrder readout = s.readOutOrder;
            if (!readout.isReadoutLeft()) {
                currentImageColumn = imagePixelData.getWidth() - 1;
                increaseImageColumnCount = false;
            } else {
                currentImageColumn = 0;
                increaseImageColumnCount = true;
            }
            scanColumnFromTop = !readout.isReadoutDown() /*&& increaseImageColumnCount*/;//WHY IS THIS???? IT FIXES p=0 ITL Segments
        }

        void fillSerialPrescan(ByteBuffer data) {
            serialPrescanBuffer.flip();
            data.put(serialPrescanBuffer);
        }

        void fillSerialOverscan(ByteBuffer data) {
            serialOverscanBuffer.flip();
            data.put(serialOverscanBuffer);
        }

        void fillParallelOverscan(ByteBuffer data) {
            parallelOverscanBuffer.flip();
            data.put(parallelOverscanBuffer);
        }

        void fillDataRow(ByteBuffer data) {
            if (!scanColumnFromTop) {
                for (int i = imagePixelData.getHeight() - 1; i >= 0; i--) {
                    int value = imagePixelData.getPixelData(currentImageColumn, i);
                    data.putInt(value);
                }
            } else {
                for (int i = 0; i < imagePixelData.getHeight(); i++) {
                    int value = imagePixelData.getPixelData(currentImageColumn, i);
                    data.putInt(value);
                }
            }
            
            currentImageColumn = increaseImageColumnCount ? currentImageColumn + 1 : currentImageColumn - 1;
        }
    }

    private ByteBuffer createByteBuffer(int size) {
        ByteBuffer dest = ByteBuffer.allocateDirect(size * 4);
        dest.order(ByteOrder.nativeOrder());
        return dest;
    }


    /**
     * Utility function to create CCDSegments by type for a given CCDType
     * and a SegmenReadOutOrder. Segments also have a channel id that is used
     * when creating fits files.
     * 
     * @param ccdType      The CCD type
     * @param readoutOrder The SegmentReadOutOrder
     * @param channel      The channel id
     *
     * @return The Segment for the provided type.
     */
    static Segment createCCDSegment(CCDType ccdType,SegmentReadOutOrder readoutOrder, int channel) {
        return new Segment(ccdType.getCCDGeometryConstants().getSegmentGeometryConstraint(),
                                readoutOrder, channel);
    }

    /**
     * 
     * Methods required by ImageSensitiveArea.
     * 
     */
    
    @Override
    public void exposeToImage(ImagePixelData imagePixelData) {
        Point absoluteOrigin = getAbsolutePoint(new Point(0, 0));
        this.imagePixelData = new SegmentPixelData(absoluteOrigin, getWidth(), getHeight(), imagePixelData);
    }

    @Override
    public boolean hasPixelData() {
        return imagePixelData != null;
    }

    @Override
    public ImagePixelData getImagePixelData() {
        return imagePixelData;
    }

    /**
     * This class extracts the Segment specific data from the overall image
     * to which this Geometry is exposed.
     * It uses the global position of this Geometry in the outermost geometry,
     * its width and height to extract the sub-region of interest.
     *
     */
    private class SegmentPixelData implements ImagePixelData {

        private final int xGlobalCoordinate, yGlobalCoordinate;
        private final int width, height;
        private final ImagePixelData exposure;

        SegmentPixelData(Point origin, int width, int height, ImagePixelData exposure) {
            this.xGlobalCoordinate = origin.x;
            this.yGlobalCoordinate = origin.y;
            this.width = width;
            this.height = height;
            this.exposure = exposure;
        }

        @Override
        public int getHeight() {
            return height;
        }

        @Override
        public int getWidth() {
            return width;
        }

        @Override
        public int getPixelData(int x, int y) {
            return exposure.getPixelData(x += xGlobalCoordinate, y += yGlobalCoordinate);
        }

        @Override
        public double getMaxValue() {
            return exposure.getMaxValue();
        }
    }

}
