package org.lsst.ccs.utilities.ccd;

import java.awt.Dimension;

/**
 * A Raft Geometry object.
 * In accordance to LCA-13501 a Raft geometry is composed of three Reb
 * geometries. A Raft object is built from the information contained in the
 * provided RaftGeometryConstants object. These constants vary depending on
 * which CCD type they correspond to.
 * 
 * We assume here that a Raft contains only the same type of CCD.
 * If we will support heterogeneous CCD types in a raft we will have to
 * figure out how to place them in the addGeometryToGrid method.
 * 
 * @author The LSST CCS Team
 */
public class Raft extends Geometry<Reb> {

    private final RaftGeometryConstants raftConstants;

    /**
     * Create a Raft object for a given type of CCD.
     * The resulting Raft is empty, having no Rebs installed.
     * 
     * @param type The type of CCD in this Raft.
     */
    public Raft(CCDType type) {
        this(RaftGeometryConstants.createRaftGeometryConstants(type));
    }
    
    
    Raft(RaftGeometryConstants raftConstants) {
        super("R", new Dimension(raftConstants.getRaftx(), raftConstants.getRafty()), 3, 1);        
        this.raftConstants = raftConstants;
    }
    
    @Override
    protected void addGeometryToGrid(Reb child, int p, int s) {
        if (s != 0) {
            throw new IllegalArgumentException("Rafts can only accept Reb children in the parallel direction.");
        }
        int xCoord = (2-p) * child.getWidth() + (2-p)*raftConstants.getGap_inx() + (int)raftConstants.getGap_outx();
        addGeometry(child, xCoord, (int)raftConstants.getGap_outy());
    }

    /**
     * Get the RaftGeometryConstants object for this Raft.
     * 
     * @return The Raft geometry constants.
     */
    public RaftGeometryConstants getRaftGeometryConstants() {
        return raftConstants;
    }
    
    /**
     * Dedicated method to add a reb to this Raft.
     * @param reb The Reb to add.
     * @param rebId The id of the reb, form 0 to 2.
     */
    public void addReb(Reb reb, int rebId) {
        addChildGeometry(reb, rebId, 0);
    }
    
    /**
     * Utility static method to create a fully Reb/CCD populated Raft for the
     * provided CCD type.
     * 
     * @param type  The type of the CCDs in the Raft.
     * @return      A fully populated Raft.
     */
    public static Raft createRaft(CCDType type) {
        Raft raft = new Raft(type);
        //Add three Rebs, right to left.
        for (int i = 2; i >= 0; i--) {
            Reb reb = Reb.createReb(type);
            raft.addReb(reb, i);
        }
        return raft;
    }
    
}
