package org.lsst.ccs.utilities.ccd;

import java.awt.Dimension;

/**
 * A Geometry class defining the Camera Focal Plane.
 * It contains a 5x5 array of Rafts. Not all of them have to be present as
 * there could be a partially filled focal plane.
 * 
 * This class needs a RaftGeometryConstants object to get the Raft Geometry
 * dimensions, both to determine the size of the FocalPlane and to place the
 * Raft children in the array.
 * 
 * Since the E2V and the ITL Rafts height and width are the same, we use 
 * chose to internally use the Raft geometry constants for an E2V CCD type.
 * We might want to revisit this choice if the above condition changes.
 * 
 * The constants are as defined in LCA-13501
 * 
 * @author The LSST CCS Team
 */
public class FocalPlane extends Geometry<Raft> {

    private static final int GAP_BETWEEN_RAFTS = 50;
    private static final int S_BAYS = 5, P_BAYS = 5;    
    private static final RaftGeometryConstants RAFT_GEOMETRY_CONSTANTS = RaftGeometryConstants.createRaftGeometryConstants(new E2VCCDType());

    /**
     * Build a Camera FocalPlane object.
     * This object has no Rafts. They need to be added.
     * 
     */
    public FocalPlane() {        
        super("FocalPlane", new Dimension(P_BAYS*RAFT_GEOMETRY_CONSTANTS.getRaftx()+(P_BAYS-1)*GAP_BETWEEN_RAFTS,S_BAYS*RAFT_GEOMETRY_CONSTANTS.getRafty()+(S_BAYS-1)*GAP_BETWEEN_RAFTS), P_BAYS , S_BAYS);
    }

    @Override
    protected void addGeometryToGrid(Raft child, int p, int s) {
        int yCoord = s * (child.getHeight()+GAP_BETWEEN_RAFTS) ;
        int xCoord = p*  (child.getWidth()+ GAP_BETWEEN_RAFTS) ;
        addGeometry(child, xCoord, yCoord);
    }

}
