package org.lsst.ccs.utilities.ccd;

/**
 * CCD Type utilities. They main purpose of this class is to help
 * change the CCDType of existing Geometries.
 * 
 * @author The LSST CCS Team
 */
public class CCDTypeUtils {
    
    /**
     * Change the type of a CCD. It will return a new CCD object for the provided
     * type if the existing ccd has a different type.
     * 
     * @param geometry The original Geometry
     * @param newType  The desired CCDType
     * @return         The Geometry for the provided type.
     */    
    public static <T extends Geometry> T changeCCDTypeForGeometry(T geometry, CCDType newType) {
        T newGeometry = createNewGeometryForCCDType(geometry, newType);        
        if ( newGeometry instanceof CCD ) {
            return newGeometry;
        }
        for ( int p = 0; p < geometry.getParallelChildrenCount(); p++ ) {
            for ( int s = 0; s < geometry.getSerialChildrenCount(); s++ ) {
                Geometry child = geometry.getChild(p, s);
                if ( child != null ) {
                    newGeometry.addChildGeometry(changeCCDTypeForGeometry(child, newType), p, s);
                }
            }
        }
        return newGeometry;
    }
    
    private static <T extends Geometry> T createNewGeometryForCCDType(T geometry, CCDType newType) {
        if ( geometry instanceof CCD ) {
            return (T)CCD.createCCD(newType);        
        } else if ( geometry instanceof Reb ) {
            return (T) new Reb(newType);
        } else if ( geometry instanceof Raft) {
            return (T) new Raft(newType);
        } else if ( geometry instanceof FocalPlane ) {
            return geometry;
        } else {
            throw new IllegalArgumentException("Cannot create new Geometry for "+geometry);
        }
    }
    
    
}
