package org.lsst.ccs.utilities.image.viewer;

import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.Image;
import java.awt.Point;
import java.awt.geom.AffineTransform;
import java.awt.image.BufferedImage;
import java.awt.image.WritableRaster;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectOutputStream;
import java.nio.ByteBuffer;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import javax.swing.JFrame;
import javax.swing.JPanel;
import javax.swing.SwingUtilities;
import org.lsst.ccs.command.annotations.Argument;
import org.lsst.ccs.command.annotations.Command;
import org.lsst.ccs.geometry.Geometry;
import org.lsst.ccs.geometry.ImageSensitiveArea;
import org.lsst.ccs.shell.JLineShell;
import org.lsst.ccs.utilities.ccd.CCD;
import org.lsst.ccs.utilities.ccd.CCDGeometry;
import org.lsst.ccs.utilities.ccd.CCDType;
import org.lsst.ccs.utilities.ccd.Raft;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.ccd.Segment;
import org.lsst.ccs.utilities.ccd.Segment.SegmentReadOutOrder;
import org.lsst.ccs.utilities.image.FitsHeaderUtilities;
import org.lsst.ccs.utilities.image.FitsFileWriter;
import org.lsst.ccs.utilities.image.FitsHeadersSpecifications;
import org.lsst.ccs.utilities.image.ImagePixelData;
import org.lsst.ccs.utilities.image.ImageSet;
import org.lsst.ccs.utilities.image.patterns.GeneratedImage;
import org.lsst.ccs.utilities.image.patterns.PatternGeneratorFactory;

/**
 * A test program to generate and visualize images with geometries.
 *
 * @author turri
 */
public class ImageWithGeometryViewer {

    private String imagePattern = "ripples";
    private final HashMap<String, String> parameters = new HashMap<>();
    private GeneratedImage generatedImage;
    private JFrame frame = new JFrame();
    private final ImagePanel imagePanel = new ImagePanel();
    private static Geometry currentObject;
    private static double scaleFactor = 0.1;

    public ImageWithGeometryViewer() {
        ImageWithGeometryViewer.setGeometry("ccd", "e2v");
        FitsHeadersSpecifications.addSpecFile("primary");
        FitsHeadersSpecifications.addSpecFile("extended");
        FitsHeadersSpecifications.addSpecFile("test_cond");
    }

    /**
     * Expose the current Geometry to a simulated image.
     *
     */
    @Command(description = "Expose the current geometry to a simulated image")
    public void exposeGeometry() {
        if (currentObject == null) {
            throw new RuntimeException("Geometry does not exist");
        }
        if (generatedImage == null) {
            generateImageForGeometry();
        }
        PatternGeneratorFactory.exposeGeometryToGeneratedImage(currentObject, generatedImage);
    }

    /**
     * Set the store Geometry to be used for generating the image.
     *
     * @param geometryName The parameter value.
     * @param sensorType The sensor type.
     */
    @Command(description = "Set the Geometry to use.")
    public static void setGeometry(
            @Argument(name = "geometry", description = "The parameter's value") String geometryName,
            @Argument(name = "type", description = "The sensor's type", defaultValue = "e2v") String sensorType) {

        switch (geometryName) {
            case "raft":
                currentObject = Raft.createRaft("Bay22(" + sensorType + ")", CCDType.valueOf(sensorType.toUpperCase()));
                break;
            case "reb":
                currentObject = Reb.createReb("Reb0(" + sensorType + ")", 0, CCDType.valueOf(sensorType.toUpperCase()));
                break;
            case "ccd":
                currentObject = CCD.createCCD("Sen00(" + sensorType + ")", CCDType.valueOf(sensorType.toUpperCase()));
                break;
            default:
                throw new IllegalArgumentException("Could not instanciate geometry " + geometryName);
        }
        System.out.println("Created Geometry " + geometryName + " of type " + sensorType);
    }

    /**
     * Set an Image generation parameter. It will be used when calling
     * the appropriate PatternGenerator to produce an image.
     *
     * @param parName The parameter name
     * @param parValue The parameter value
     */
    @Command(description = "Set an Image generation parameter.")
    public void setImageGenerationParameter(
            @Argument(name = "parName", description = "The parameter's name") String parName,
            @Argument(name = "parValue", description = "The parameter's value") String parValue) {
        parameters.put(parName, parValue);
    }

    /**
     * List the patterns that are available for generating the Image.
     *
     * @return The Set of the available patterns.
     */
    @Command
    public Set<String> listAvailablePatterns() {
        return PatternGeneratorFactory.listAvailablePatterns();
    }

    protected enum SetGetImagePattern {

        IMAGE_PATTERN
    };

    @Command(description = "Set the image pattern. This is used to generate images.")
    public void set(@Argument(name = "item") SetGetImagePattern what, @Argument(name = "value", defaultValue = "") String value) {
        switch (what) {
            case IMAGE_PATTERN:
                if (listAvailablePatterns().contains(value)) {
                    imagePattern = value;
                } else {
                    throw new RuntimeException("Pattern \"" + value + "\" is not supported.");
                }
        }
    }

    @Command(description = "Get various settings")
    public String get(@Argument(name = "item") SetGetImagePattern what) {
        switch (what) {
            case IMAGE_PATTERN:
                return imagePattern;
            default:
                return null;
        }
    }

    @Command(description = "Show the geometry tree.")
    public void showGeometryTree() {
        if (currentObject == null) {
            throw new RuntimeException("There is currently no geometry availabble.");
        }
        printGeometryTree(currentObject, "", new Point(0, 0));

    }

    private static void printGeometryTree(Geometry geom, String indent, Point p) {
        System.out.println(indent + "-> " + geom.getName() + " at " + p.x + "," + p.y);
        if (geom.hasChildren()) {
            Map<Geometry, Point> children = geom.getChildrenWithAbsoluteCoordinates();
            for (Geometry child : children.keySet()) {
                printGeometryTree(child, indent + "---", children.get(child));
            }
        }
    }

    //TO-DO What is this meant to do?
//    @Command(description = "Generate an image based on the current pattern and generation parameters.")
//    public void generateImage(
//            @Argument(name = "width", defaultValue = "220", description = "The Image width in pixels") int width,
//            @Argument(name = "height", defaultValue = "370", description = "The Image height in pixels") int height) {
//
//        Geometry g = new Geometry("test", width, height);
//        generatedImage = PatternGeneratorFactory.generateImageForGeometry(g, imagePattern, parameters);
//
//    }

    @Command(description = "Generate an image for the currently loaded Geometry.")
    public void generateImageForGeometry() {
        generatedImage = PatternGeneratorFactory.generateImageForGeometry(currentObject, imagePattern, parameters);
    }

    @Command(description = "Show the generated Image.")
    public void showImage() throws Exception {
        if (generatedImage == null) {
            throw new RuntimeException("No image has been generated.");
        }
        SwingUtilities.invokeAndWait(new Runnable() {

            @Override
            public void run() {
                imagePanel.setImage(convertGeneratedImageForDisplay(generatedImage));
                if (frame.getContentPane() != imagePanel) {
                    frame.setContentPane(imagePanel);
                }
                if (frame.isVisible()) {
                    frame.revalidate();
                    frame.repaint();
                    frame.pack();
                } else {
                    frame.pack();
                    frame.setVisible(true);
                }
            }

        });
    }

    @Command(description = "Show the geometry and its data if present.")
    public void showGeometry() throws Exception {

        SwingUtilities.invokeAndWait(new Runnable() {

            @Override
            public void run() {
                imagePanel.setImage(createBufferedImageForGeometry(currentObject));
                imagePanel.setGeometry(currentObject);
                if (frame.getContentPane() != imagePanel) {
                    frame.setContentPane(imagePanel);
                }
                if (frame.isVisible()) {
                    frame.revalidate();
                    frame.repaint();
                    frame.pack();
                } else {
                    frame.pack();
                    frame.setVisible(true);
                }
            }

        });
    }

    @Command(description = "Show the data streams.")
    public void showDataStreams() throws Exception {

        SwingUtilities.invokeAndWait(new Runnable() {

            @Override
            public void run() {
                imagePanel.setImage(createBufferedImageForGeometryFromDataStreams(currentObject));
//                if ( currentObject instanceof CCD) {
//                    imagePanel.setGeometry(((CCD)currentObject).getExpandedGeometry());                    
//                } else {
                imagePanel.setGeometry(currentObject);
//                }

                if (frame.getContentPane() != imagePanel) {
                    frame.setContentPane(imagePanel);
                }
                if (frame.isVisible()) {
                    frame.revalidate();
                    frame.repaint();
                    frame.pack();
                } else {
                    frame.pack();
                    frame.setVisible(true);
                }
            }

        });
    }

    @Command(description = "Write the fits files with the accumulated images")
    public void writeFitsFiles() throws Exception {

        writeFitsFilesForObject(currentObject);

    }

    private CCDGeometry.ReadoutOrder convert(SegmentReadOutOrder srop) {
        if (srop.isReadoutDown()) {
            if (srop.isReadoutLeft()) {
                return CCDGeometry.ReadoutOrder.UpLeft;
            } else {
                return CCDGeometry.ReadoutOrder.DownLeft;
            }
        } else if (srop.isReadoutLeft()) {
            return CCDGeometry.ReadoutOrder.UpRight;
        } else {
            return CCDGeometry.ReadoutOrder.DownRight;
        }
    }

    private void writeFitsFilesForObject(Object obj) throws Exception {
        if (obj instanceof CCD) {

            CCD ccd = (CCD) obj;
            ImageSet imageSet = FitsHeaderUtilities.createImageSetForCCD(ccd);

            String raftName = String.format("R99_S00");
            File raftFile = new File("/tmp/" + raftName + ".fits");
            int channel = 0;
            try (FitsFileWriter ffw = new FitsFileWriter(raftFile, imageSet)) {
                for (int s = 0; s < ccd.getSerialChildrenCount(); s++) {
                    for (int p = 0; p < ccd.getParallelChildrenCount(); p++) {
                        Segment seg = ccd.getChild(p, s);
                        ByteBuffer dataBuffer = seg.getRawImageData().getImageData();
                        ffw.write(channel, dataBuffer);
                        dataBuffer.clear();
                        channel++;
                    }
                }
            }

        }

    }

    private BufferedImage createBufferedImageForGeometryFromDataStreams(Geometry geom) {
        if (geom instanceof CCD) {
            CCD ccd = (CCD) geom;
            BufferedImage result = new BufferedImage(ccd.getType().getCCDpx(), ccd.getType().getCCDpy(), BufferedImage.TYPE_BYTE_GRAY);
            addGeometryDataStreamToRaster(new Point(0, 0), result.getRaster(), geom);
            return result;
        } else {
            throw new RuntimeException("Currently can only produce images from CCDs.");
        }
    }

    private void addGeometryDataStreamToRaster(Point origin, WritableRaster raster, Geometry geom) {
        
        //If we want to see the DataStream correctly, we have to flip the data depending
        //on the Segment's readout point.
        
        if (geom instanceof CCD) {
            CCD ccd = (CCD) geom;
            for (int p = 0; p < ccd.getParallelChildrenCount(); p++) {
                for (int s = 0; s < ccd.getSerialChildrenCount(); s++) {

                    Segment seg = ccd.getChild(p, s);
                    Point segmentPoint = ccd.getGeometryPoint(seg);
                    double maxValue = seg.getImagePixelData().getMaxValue();

                    ByteBuffer bb = seg.getRawImageData().getImageData();

                    for (int pp = 0; pp < seg.getSegmentParallelActiveSize(); pp++) {
                        for (int j = 0; j < seg.getSegmentSerialPrescanSize(); j++) {
                            bb.getInt();
                        }
                        for (int ss = 0; ss < seg.getSegmentSerialActiveSize(); ss++) {
                            int val = (int) (256 * bb.getInt() / maxValue);
                            raster.setSample(segmentPoint.x + pp, segmentPoint.y + ss, 0, val);
                        }
                        for (int j = 0; j < seg.getSegmentSerialOverscanSize(); j++) {
                            bb.getInt();
                        }
                    }

                }
            }
        } else {
            throw new RuntimeException("Currently can only produce images from CCDs.");
        }
    }

    private final BufferedImage createBufferedImageForGeometry(Geometry geom) {
        int width = geom.getWidth();
        int height = geom.getHeight();
        BufferedImage result = new BufferedImage(width, height, BufferedImage.TYPE_BYTE_GRAY);
        addGeometryToRaster(new Point(0, 0), result.getRaster(), geom);
        return result;
    }

    private void addGeometryToRaster(Point origin, WritableRaster raster, Geometry<?> geom) {
        if (geom instanceof ImageSensitiveArea) {
            ImageSensitiveArea imageSensitiveArea = (ImageSensitiveArea) geom;
            if (imageSensitiveArea.hasPixelData()) {
                addDataToRaster(origin, raster, imageSensitiveArea.getImagePixelData());
            }
        }
        for (Geometry child : geom.getChildrenList()) {
            addGeometryToRaster(child.getGeometryAbsolutePosition(), raster, child);
        }
    }

    private final BufferedImage convertGeneratedImageForDisplay(GeneratedImage generatedImage) {
        int width = generatedImage.getWidth();
        int height = generatedImage.getHeight();
        BufferedImage result = new BufferedImage(width, height, BufferedImage.TYPE_BYTE_GRAY);
        addDataToRaster(new Point(0, 0), result.getRaster(), generatedImage);
        return result;
    }

    private final void addDataToRaster(Point origin, WritableRaster raster, ImagePixelData data) {
        double maxValue = data.getMaxValue();
        if (Double.isNaN(maxValue)) {
            maxValue = 1.;
        }
        int width = data.getWidth();
        int height = data.getHeight();

        for (int x = 0; x < width; x++) {
            int xGlobal = x + origin.x;
            for (int y = 0; y < height; y++) {
                int yGlobal = y + origin.y;
                raster.setSample(xGlobal, yGlobal, 0, (int) (256 * (data.getPixelData(x, y)) / maxValue));
            }
        }

    }

    @Command(description = "Save the generated image to file")
    public void saveGeneratedImage(
            @Argument(name = "fullPath", defaultValue = "/tmp/generateImage.ser") String fullPath) {

        if (generatedImage == null) {
            throw new RuntimeException("No image has been generated.");
        }

        String format = fullPath.substring(fullPath.lastIndexOf(".") + 1);
        System.out.println("Saving image to " + fullPath + " format " + format);

        try {
            File outputfile = new File(fullPath);
            if (format.equals("ser")) {
                ObjectOutputStream obj_out = new ObjectOutputStream(new FileOutputStream(outputfile));
                obj_out.writeObject(generatedImage);
            }
        } catch (IOException e) {
            throw new RuntimeException("Could not save generated image.", e);
        }
    }
    
    @Command(description = "Set the scale factor")
    public void setScaleFactor(@Argument(name = "scaleFactor", defaultValue = "0.1") double scaleFactor) {
        this.scaleFactor = scaleFactor;
    }


    
    private static class ImagePanel extends JPanel {

        Image i;
        Geometry geom;
        int width = 0, height = 0;

        @Override
        protected void paintComponent(Graphics g) {
            Graphics2D g2d = (Graphics2D)g;
            g2d.scale(scaleFactor,scaleFactor);
            super.paintComponent(g);
            g.drawImage(i, 0, 0, null);
            
            if (geom != null) {
                GeometryRenderer.renderGeometry(currentObject, g, new Point(0, 0),scaleFactor);
            }
        }

        public void setImage(Image i) {
            this.i = i;
            updateDimensions();
        }

        public void setGeometry(Geometry g) {
            this.geom = g;
            updateDimensions();
        }

        private void updateDimensions() {
            width = i.getWidth(null);
            height = i.getHeight(null);
            if (geom != null) {
                if (geom.getWidth() > width) {
                    width = geom.getWidth();
                }
                if (geom.getHeight() > height) {
                    height = geom.getHeight();
                }
            }
        }

        @Override
        public Dimension getPreferredSize() {
            return new Dimension((int)(scaleFactor*width), (int)(scaleFactor*height));
        }

    }

    public static void main(String[] arg) throws Exception {
        String[] args = new String[]{"-dc", "org.lsst.ccs.utilities.image.patterns.TestImageWithGeometryGeneration"};
        JLineShell.main(args);
    }

}
