
package org.lsst.ccs.utilities.image.patterns;

import java.util.Map;

/**
 * A PatternGenerator that creates ripples extending out from a center with constant
 * amplitude and frequency over a mean baseline.
 * 
 * The allowed parameters for this patternGenerator are:
 *   - ripples_xCenter: the x coordinate of the center of the ripples 
 *   - ripples_yCenter: the y coordinate of the center of the ripples 
 *   - ripples_amplitude: the amplitude of the ripples 
 *   - ripples_mean: the mean level of the ripples around which the amplitude fluctuates
 *   - ripples_frequency: the ripples frequency
 * 
 * @author turri
 */
public class RipplesPatternGenerator extends PatternGenerator {
    
    RipplesPatternGenerator() {
        super("ripples");
    }

    @Override
    public GeneratedImage generateImage(int width, int height, Map<String, String> parameters) {

        double xCenter = getDoubleParameter("ripples_xCenter", parameters, (double) width / 3.);
        double yCenter = getDoubleParameter("ripples_yCenter", parameters, (double) height / 7.);
        
        double amplitude = getDoubleParameter("ripples_amplitude", parameters, 200);
        double mean = getDoubleParameter("ripples_mean", parameters, 1000);
        double frequency = getDoubleParameter("ripples_frequency", parameters, 30);
        
        int[][] data = new int[width][height];
        for (int i = 0; i < width; i++) {
            double dx = (double) i - xCenter;
            for (int j = 0; j < height; j++) {
                double dy = (double) j - yCenter;
                double r = Math.sqrt(Math.pow(dy, 2) + Math.pow(dx, 2));
                data[i][j] = (int) (mean + amplitude * Math.sin(r / frequency));
            }
        }
        return new GeneratedImage(data, mean+amplitude);        
    }
    
}
