
package org.lsst.ccs.utilities.image;

import java.util.Map;

/**
 * An interface to provide Metadata to fill the content of Fits Headers.
 * According to the syntax used in header specification (spec) files the metadata
 * is in the form of a Map of Maps: the key names for the maps are defined as "mapName"
 * and are used to determine at runtime the value of a fits keyword defined in the
 * spec file as ${"mapName"."mapKey"}. The fits keyword is assigned the value
 * corresponding to key "mapKey" in the map stored with key "mapName".
 * 
 * If "mapName" is not specified (i.e. ${"mapKey"}) then all the maps are searched
 * and the first one found containing key "mapKey" will yield the value.
 * 
 * This interface is designed to be able to provide all the metadata needed to 
 * write a single fits file. 
 * 
 * Since a single fit file is meant to be written for
 * each CCD, the primary header metadata should contain information for the CCD
 * itself, the Reb used to read it, the Raft in which it is embedded etc.
 * The extended data header should be written for each of the CCD segments, containing
 * information for the specific segment.
 * 
 * @author turri
 */
public interface FitsHeaderMetadataProvider {
    
    /**
     * Get the metadata for the primary Header
     * 
     * @return The metadata for the primary header
     */
    public Map<String,Map<String,Object>> getPrimaryHeaderMetadata();
    
    /**
     * Get the metadata for the extended Headers with data by index.
     * These are the headers containing the actual images.
     * 
     * @param extendedIndex The index of the extended header.
     * 
     * @return The metadata for the requested extended header
     */
    public Map<String,Map<String,Object>> getDataExtendedHeaderMetadata(int extendedIndex);
    
    /**
     * Get the metadata for an additional extended Headers by key. The information
     * in these extended headers is stored as BinaryTables. These are all the
     * headers defined in the spec files with keywords other than "primary" and
     * "extended"; this keyword will be used in this method to identify which
     * metadata map to return.
     * 
     * @param extendedKeyword The keyword identifying the extended header.
     * 
     * @return The metadata for the requested extended header
     */
    public Map<String,Map<String,Object>> getAdditionalExtendedHeaderMetadata(String extendedKeyword);

}
