package org.lsst.ccs.utilities.ccd;

import java.awt.Dimension;
import java.awt.Point;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import org.lsst.ccs.geometry.Geometry;
import org.lsst.ccs.geometry.ImageSensitiveArea;
import org.lsst.ccs.utilities.ccd.image.data.RawImageDataProvider;
import org.lsst.ccs.utilities.ccd.image.data.RawImageData;
import org.lsst.ccs.utilities.image.ImagePixelData;

/**
 * An class to represent a CCD Segment.
 *
 * @author turri
 */
public class Segment extends Geometry implements RawImageDataProvider, ImageSensitiveArea, SegmentInterface {

    private final int serialPrescan, serialOverscan, parallelOverscan;
    private final SegmentReadOutOrder readoutPoint;
    private ImagePixelData imagePixelData;
    private final int serialActive, parallelActive;
    private final int serialPosition, parallelPosition;
    private final int channel;
    private final CCDType ccdType;

    /**
     * CCDSegment constructor.
     *
     * @param serialActive Length of the portion of the serial register that receives signal from the array.
     * @param parallelActive Number of light sensitive rows in the CCDSegment.
     * @param serialPrescan Number of prescan pixels in the serial register.
     * @param serialOverscan Number of additional phantom pixels that are read out from the serial register.
     * @param parallelOverscan Number of additional phantom rows that are read from the CCDSegment.
     */
    Segment(CCDType ccdType, String name, int serialPosition, int parallelPosition, int channel) {
        super(name, new Dimension(ccdType.getSegmentParallelActiveSize(),ccdType.getSegmentSerialActiveSize()));
        this.serialPrescan = ccdType.getSegmentSerialPrescanSize();
        this.serialOverscan = ccdType.getSegmentSerialOverscanSize();
        this.parallelOverscan = ccdType.getSegmentParallelOverscanSize();
        this.serialActive = ccdType.getSegmentSerialActiveSize();
        this.parallelActive = ccdType.getSegmentParallelActiveSize();
        this.serialPosition = serialPosition;
        this.parallelPosition = parallelPosition;
        if (parallelPosition == 1) {
            readoutPoint = SegmentReadOutOrder.LeftUp;
        } else {
            readoutPoint = ccdType.name().toLowerCase().contains("e2v") ? SegmentReadOutOrder.RightDown : SegmentReadOutOrder.RightUp;
        }
        this.channel = channel;
        this.ccdType = ccdType;
    }
    
    public CCDType getCCDType() {
        return ccdType;
    }

    void setRedout(boolean isTopRow, String CCDtype) {
        if (readoutPoint != null) {
            throw new RuntimeException("The Segment Readout point can only be set once!");
        }
    }

    @Override
    public int getChannel() {
        return channel;
    }

    @Override
    public int getParallelPosition() {
        return parallelPosition;
    }

    @Override
    public int getSerialPosition() {
        return serialPosition;
    }

    public SegmentReadOutOrder getSegmentReadOutPoint() {
        return readoutPoint;
    }

    @Override
    public int getSegmentSerialPrescanSize() {
        return serialPrescan;
    }

    @Override
    public int getSegmentSerialOverscanSize() {
        return serialOverscan;
    }

    @Override
    public int getSegmentParallelOverscanSize() {
        return parallelOverscan;
    }

    @Override
    public int getSegmentSerialActiveSize() {
        return serialActive;
    }

    @Override
    public int getSegmentParallelActiveSize() {
        return parallelActive;
    }

    @Override
    public int getSegmentTotalSerialSize() {
        return getSegmentSerialPrescanSize() + getSegmentSerialActiveSize() + getSegmentSerialOverscanSize();
    }

    @Override
    public int getSegmentTotalParallelSize() {
        return getSegmentParallelActiveSize() + getSegmentParallelOverscanSize();
    }

    @Override
    public boolean isReadoutDown() {
        return readoutPoint.isReadoutDown();
    }

    @Override
    public boolean isReadoutLeft() {
        return readoutPoint.isReadoutLeft();
    }

    @Override
    public CCDInterface getParentCCD() {
        return (CCDInterface)getParent();
    }

    
    
    /**
     * Get the buffer of image data accumulated by this Segment.
     *
     * @return The ByteBuffer with the data;
     */
    @Override
    public RawImageData getRawImageData() {
        DataStreamer streamer = new DataStreamer(this);
        ByteBuffer dest = createByteBuffer(getSegmentTotalSerialSize() * getSegmentTotalParallelSize());
        for (int row = 0; row < getSegmentParallelActiveSize(); row++) {
            streamer.fillSerialPrescan(dest);
            streamer.fillDataRow(dest);
            streamer.fillSerialOverscan(dest);
        }
        streamer.fillParallelOverscan(dest);
        dest.flip();
        return new RawImageData(RawImageData.BitsPerPixel.BIT32, dest);
    }

    
    /**
     * This class has to provide the readout bytes in the order in which the
     * segments are read out. So it has to convert a rectangular image with
     * (0,0) in the top left corner to a stream corresponding to the data flow.
     * The data in the rectangle has to be ordered appropriately depending on the
     * segment's ReadOutPoint value:
     * 
     * - if the readout is to the right the image rectangle has to be read from left
     *   to right, otherwise from left to right.
     * - if the readout is down, the image rectangle has to be read top to bottom,
     *   otherwise bottom to top.
     *  
     */
    private class DataStreamer {

        private final ByteBuffer serialPrescanBuffer;
        private final ByteBuffer serialOverscanBuffer;
        private final ByteBuffer parallelOverscanBuffer;
        private final ImagePixelData imagePixelData;
        private int currentImageColumn;
        private boolean increaseImageColumnCount;
        private boolean scanColumnFromTop;
        private final Segment s;

        DataStreamer(Segment s) {
            serialPrescanBuffer = createByteBuffer(s.getSegmentSerialPrescanSize());
            serialOverscanBuffer = createByteBuffer(s.getSegmentSerialOverscanSize());
            parallelOverscanBuffer = createByteBuffer(s.getSegmentParallelOverscanSize() * s.getSegmentTotalSerialSize());
            for (int i = 0; i < s.getSegmentSerialPrescanSize(); i++) {
                serialPrescanBuffer.putInt(0);
            }
            for (int i = 0; i < s.getSegmentSerialOverscanSize(); i++) {
                serialOverscanBuffer.putInt(0);
            }
            for (int i = 0; i < s.getSegmentParallelOverscanSize() * s.getSegmentTotalSerialSize(); i++) {
                parallelOverscanBuffer.putInt(0);
            }
            imagePixelData = s.getImagePixelData();

            this.s = s;
            reset();
        }

        public final void reset() {
            
            
            SegmentReadOutOrder readout = s.readoutPoint;
            if (!readout.isReadoutLeft()) {
                currentImageColumn = imagePixelData.getWidth() - 1;
                increaseImageColumnCount = false;
            } else {
                currentImageColumn = 0;
                increaseImageColumnCount = true;
            }
            scanColumnFromTop = !readout.isReadoutDown() /*&& increaseImageColumnCount*/;//WHY IS THIS???? IT FIXES p=0 ITL Segments
        }

        void fillSerialPrescan(ByteBuffer data) {
            serialPrescanBuffer.flip();
            data.put(serialPrescanBuffer);
        }

        void fillSerialOverscan(ByteBuffer data) {
            serialOverscanBuffer.flip();
            data.put(serialOverscanBuffer);
        }

        void fillParallelOverscan(ByteBuffer data) {
            parallelOverscanBuffer.flip();
            data.put(parallelOverscanBuffer);
        }

        void fillDataRow(ByteBuffer data) {
            if (!scanColumnFromTop) {
                for (int i = imagePixelData.getHeight() - 1; i >= 0; i--) {
                    int value = imagePixelData.getPixelData(currentImageColumn, i);
                    data.putInt(value);
                }
            } else {
                for (int i = 0; i < imagePixelData.getHeight(); i++) {
                    int value = imagePixelData.getPixelData(currentImageColumn, i);
                    data.putInt(value);
                }
            }
            
            currentImageColumn = increaseImageColumnCount ? currentImageColumn + 1 : currentImageColumn - 1;
        }
    }

    private ByteBuffer createByteBuffer(int size) {
        ByteBuffer dest = ByteBuffer.allocateDirect(size * 4);
        dest.order(ByteOrder.nativeOrder());
        return dest;
    }

    /**
     * The Segment readout point with respect to the Camera Coordinate System.
     * The canonical readout point when dealing with fits files is LeftDown; any
     * other readout point will need some sort of transformation.
     *
     */
    public enum SegmentReadOutOrder {

        LeftDown(true, true), LeftUp(true, false), RightDown(false, true), RightUp(false, false);

        private final boolean isLeft, isDown;

        private SegmentReadOutOrder(boolean isLeft, boolean isDown) {
            this.isLeft = isLeft;
            this.isDown = isDown;
        }

        public boolean isReadoutDown() {
            return isDown;
        }

        public boolean isReadoutLeft() {
            return isLeft;
        }

    };

    /**
     * Utility function to create CCDSegments by type.
     *
     * @param ccdType The type of the CCD.
     * @param label The label of the segment.
     *
     * @return The Segment for the provided type.
     */
    public static Segment createCCDSegment(CCDType ccdType, String label, int serialPosition, int parallelPosition, int channel) {
        return new Segment(ccdType, label, serialPosition, parallelPosition, channel);
    }

    @Override
    public void exposeToImage(ImagePixelData imagePixelData) {
        Point absoluteOrigin = getAbsolutePoint(new Point(0, 0));
        this.imagePixelData = new SegmentPixelData(absoluteOrigin, getWidth(), getHeight(), imagePixelData);
    }

    @Override
    public boolean hasPixelData() {
        return imagePixelData != null;
    }

    @Override
    public ImagePixelData getImagePixelData() {
        return imagePixelData;
    }

    /**
     * This class extracts the Segment specific data from the overall image
     * to which this Geometry is exposed.
     * It uses the global position of this Geometry in the outermost geometry,
     * its width and height to extract the sub-region of interest.
     *
     */
    private class SegmentPixelData implements ImagePixelData {

        private final int xGlobalCoordinate, yGlobalCoordinate;
        private final int width, height;
        private final ImagePixelData exposure;

        SegmentPixelData(Point origin, int width, int height, ImagePixelData exposure) {
            this.xGlobalCoordinate = origin.x;
            this.yGlobalCoordinate = origin.y;
            this.width = width;
            this.height = height;
            this.exposure = exposure;
        }

        @Override
        public int getHeight() {
            return height;
        }

        @Override
        public int getWidth() {
            return width;
        }

        @Override
        public int getPixelData(int x, int y) {
            return exposure.getPixelData(x += xGlobalCoordinate, y += yGlobalCoordinate);
        }

        @Override
        public double getMaxValue() {
            return exposure.getMaxValue();
        }
    }

}
