package org.lsst.ccs.utilities.ccd;

import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.utilities.ccd.CCDGeometry.ReadoutOrder;

/**
 * An enumeration of standard camera CCD types. Can be used to get the geometry of a CCD.
 * @author tonyj
 */
public enum CCDType {

    E2V(10, 2002, 512, 22, 46, true, 96.5, 52,28,25,26.5,25), ITL(3, 2000, 509, 32, 48, false, 99, 63,27,27,26,26);
    
    private CCDGeometry ccdGeom;
    private static final int SEGMENTS_ALONG_SERIAL = 8;
    private static final int SEGMENTS_ALONG_PARALLEL = 2;
    private final int serialActiveSize, parallelActiveSize, serialPrescanSize, serialOverscanSize, parallelOverscanSize;
    private final double parallelEdge, serialEdge;
    private final int gap_inx, gap_iny;
    private final double gap_outx, gap_outy;

    
    /**
     * CCD quantities according to LCA-13501
     * @param prescan
     * @param segmentWidth   Number of pixels in the vertical direction    dimv
     * @param segmentHeight  Number of pixels in the horizontal direction  dimh
     * @param overscan
     * @param parallelOverscan
     * @param isE2V
     * @param parallelEdge   The CCD edge along each parallel size in pixels
     * @param serialEdge     The CCD edge along each serial size in pixels
     * @param gap_inx        The inter CCD gap along the parallel size in pixels
     * @param gap_iny        The inter CCD gap along the serial size in pixels
     * @param gap_outx       The gaps at edges of raft along the parallel size in pixels
     * @param gap_outy       The gaps at edges of raft along the serial size in pixels
     * 
     */
    private CCDType(int prescan, int segmentWidth, int segmentHeight,
            int overscan, int parallelOverscan, boolean isE2V, double parallelEdge, double serialEdge, int gap_inx, int gap_iny, double gap_outx, double gap_outy) {        
        this.serialActiveSize = segmentHeight;
        this.parallelActiveSize = segmentWidth;
        this.serialPrescanSize = prescan;
        this.serialOverscanSize = overscan;
        this.parallelOverscanSize = parallelOverscan;
        this.parallelEdge = parallelEdge;
        this.serialEdge = serialEdge;
        this.gap_inx = gap_inx;
        this.gap_iny = gap_iny;
        this.gap_outx = gap_outx;
        this.gap_outy = gap_outy;

        ccdGeom = new CCDGeometry(SEGMENTS_ALONG_SERIAL, SEGMENTS_ALONG_PARALLEL, segmentHeight, segmentWidth, prescan,
                overscan, parallelOverscan, this);
        
        for (int i = 0; i < 8; i++) {
            ccdGeom.addSegment(i + 1, i, 1, ReadoutOrder.UpLeft);
        }
        for (int i = 8; i < 16; i++) {
            ccdGeom.addSegment(i + 1, 15-i, 0, isE2V ? ReadoutOrder.DownRight : ReadoutOrder.UpRight);
        }
    }
    
    
    /**
     * Methods to get Designator variables as specified in LCA-13501 Table 8.
     */

    /**
     * Get the Segment parallel active size.
     * @return The Segment parallel active size.
     */
    public int getSegmentParallelActiveSize() {
        return getDimv();
    }
    public int getDimv() {
        return parallelActiveSize;
    }
    
    /**
     * Get the Segment serial active size.
     * @return The Segment serial active size.
     */
    public int getSegmentSerialActiveSize() {
        return getDimh();
    }    
    public int getDimh() {
        return serialActiveSize;
    }
    
    /**
     * The CCD dimension, in pixels, of the active area along the parallel direction.
     * This is the segments parallel size multiplied by the number of segments
     * laid out along the parallel direction.
     * @return The CCD dimension, in pixels, of the active area along the parallel direction.
     */
    public int getActiveAreaParallelSize() {
        return getCCDax();
    }
    public int getCCDax() {
        return getDimv()*SEGMENTS_ALONG_PARALLEL;
    }
    
    /**
     * The CCD dimension, in pixels, of the active area along the serial direction.
     * This is the segments serial size multiplied by the number of segments
     * laid out along the serial direction.
     * @return The CCD dimension, in pixels, of the active area along the serial direction.
     */
    public int getActiveAreaSerialSize() {
        return getCCDay();
    }
    public int getCCDay() {
        return getDimh()*SEGMENTS_ALONG_SERIAL;
    }
    
    
    /**
     * The CCD dimension, in pixels, of the physical area along the parallel direction.
     * This is the sum of the segments parallel size multiplied by the number of segments
     * laid out along the parallel direction and twice the CCD edge on the parallel direction.
     * @return The CCD dimension, in pixels, of the physical area along the parallel direction.
     */
    public int getPhysicalAreaParallelSize() {
        return getCCDpx();
    }
    public int getCCDpx() {
        return getCCDax() + (int)(2*parallelEdge);
    }
    
    /**
     * The CCD dimension, in pixels, of the physical area along the serial direction.
     * This is the sum of the segments serial size multiplied by the number of segments
     * laid out along the serial direction and twice the CCD edge on the serial direction.
     * @return The CCD dimension, in pixels, of the physical area along the serial direction.
     */
    public int getPhysicalAreaSerialSize() {
        return getCCDpy();
    }
    public int getCCDpy() {
        return getCCDay() + (int)(2*serialEdge);
    }
    
    public double getParallelEdge() {
        return parallelEdge;
    }

    public double getSerialEdge() {
        return serialEdge;
    }
    
    public int getGap_inx() {
        return gap_inx;
    }

    public int getGap_iny() {
        return gap_iny;
    }
    
    public double getGap_outx() {
        return gap_outx;
    }

    public double getGap_outy() {
        return gap_outy;
    }
    
    public int getRaftx() {
        return 3*getCCDpx() + 2*getGap_inx() + (int)(2 * getGap_outx());
    }
    
    public int getRafty() {
        return 3*getCCDpy() + 2*getGap_iny() + (int)(2 * getGap_outy());
    }
    /**
     * Get the Segment total serial size: prescan, overscan and active size.
     * @return The Segment total serial size.
     */
    public int getSegmentTotalSerialSize() {
        return getSegmentSerialActiveSize()+getSegmentSerialOverscanSize()+getSegmentSerialPrescanSize();
    }

    /**
     * Get the Segment serial prescan size.
     * @return The Segment serial prescan size.
     */
    public int getSegmentSerialPrescanSize() {
        return serialPrescanSize;
    }
    public int getPreh() {
        return getSegmentSerialPrescanSize();
    }

    /**
     * Get the Segment serial overscan size.
     * @return The Segment serial overscan size.
     */
    public int getSegmentSerialOverscanSize() {
        return serialOverscanSize;
    }
    public int getOverh() {
        return getSegmentSerialOverscanSize();
    }

    /**
     * Get the Segment total parallel size: overscan and active size.
     * @return The Segment total parallel size.
     */
    public int getSegmentTotalParallelSize() {
        return getSegmentParallelActiveSize()+getSegmentParallelOverscanSize();
    }

    /**
     * Get the Segment parallel overscan size.
     * @return The Segment parallel overscan size.
     */
    public int getSegmentParallelOverscanSize() {
        return parallelOverscanSize;
    }    
    public int getOverv() {
        return getSegmentParallelOverscanSize();
    }
    
    /**
     * Get the geometry for this CCD.
     * @return The geometry corresponding to this CCD type.
     */
    public CCDGeometry getGeometry() {
        return ccdGeom;
    }

};
