package org.lsst.ccs.utilities.ccd;

/**
 * Allows transforms between CCD coordinate system and CCD segment (amplifier) 
 * coordinate systems. The CCD coordinate system uses the conventions of the camera coordinate
 * system (confusingly called CCS).
 * Any set method may be used to set the currently selected coordinate, and all get 
 * methods will then return coordinates corresponding to the selected coordinate. The
 * precise meaning of the x,y coordinates depends on which method was used to obtain
 * the transform.
 * @see CCDGeometry#getGlobalTransform() 
 * @see CCDGeometry#getActiveTransform() 
 * @author tonyj
 */
public interface CCDTransform {

    /**
     * Enumerate the different types of pixel that can be read out.
     */
    public enum PixelType {
        /**
         * An active (signal) pixel 
         */
        ACTIVE, SERIAL_PRESCAN, SERIAL_OVERSCAN, PARALLEL_OVERSCAN}
    /**
     * Select the absolute x,y CCD coordinate.
     * @param x The x coordinate in the CCD coordinate system
     * @param y The u coordinate in the CCD coordinate system
     */
    void setXY(int x, int y);
    /**
     * Get the CCD x coordinate
     * @return The selected CCD x coordinate
     */
    int getX();
    /**
     * Get the CCD selected y coordinate system
     * @return The CCD y coordinate
     */
    int getY();
    /**
     * Set the current segment, and serial and parallel pixel position within
     * the segment (in segment readout order).
     * @param segment The CCD segment (amplifier)
     * @param serial The serial coordinate within the CCD segment
     * @param parallel The parallel coordinate within the CCD segment
     */
    void setSegmentSerialParallel(SegmentInterface segment, int serial, int parallel);
    /**
     * The CCD segment corresponding to the currently selected coordinate.
     * @return The segment
     */
    SegmentInterface getSegment();
    /**
     * The serial position of the currently selected pixel within the current CCD segment.
     * @return The serial coordinate
     */
    int getSerial();
    /**
     * The parallel coordinate of the currently selected pixel within the current CCD segment.
     * @return The parallel coordinate
     */
    int getParallel();
    /**
     * The type of pixel corresponding to the currently selected coordinate.
     * @return The pixel type
     */
    PixelType getPixelType();
    /**
     * The global coordinate corresponding to the currently selected coordinate. 
     * If the CCDTransform was obtained via {@link CCDGeometry#getGlobalTransform()} then 
     * it will be equivalent the the x coordinate, but if it was obtained via 
     * {@link CCDGeometry#getActiveParallelSize()} it will return the global x coordinate
     * corresponding to the currently selected active pixel.
     * @return The global x coordinate
     */
    int getGlobalX();
    /**
     * The global coordinate corresponding to the currently selected coordinate. 
     * If the CCDTransform was obtained via {@link CCDGeometry#getGlobalTransform()} then 
     * it will be equivalent the the y coordinate, but if it was obtained via 
     * {@link CCDGeometry#getActiveParallelSize()} it will return the global y coordinate
     * corresponding to the currently selected active pixel.
     * @return The global y coordinate
     */ 
    int getGlobalY();
    
    /**
     * Create a CCDTransform which considers all pixels for the absolute x,y
     * coordinates.
     *
     * @param ccd The CCD on which this transformation acts.
     * @return The CCDTransform for global coordinates
     */
    public static CCDTransform getGlobalTransform(CCDInterface ccd) {
        return new GlobalTransform(ccd);
    } 

    /**
     * Create a CCDTransform which only considers active pixels for the absolute
     * x,y coordinates. When using this transform PixelType will always be
     * active. Setting a CCD coordinate which does not correspond to an active
     * pixel will result in an error. The {@link CCDTransform#getGlobalX()} and
     * {@link CCDTransform#getGlobalY()} methods can be used to obtain the
     * global coordinate corresponding to the x,y given.
     *
     * @param ccd The CCD on which this transformation acts.
     * @return The CCDTransform for active coordinates.
     */
    public static CCDTransform getActiveTransform(CCDInterface ccd) {
        return new ActiveTransform(ccd);
    }

    
    /**
     * A CCDTransform which only considers active pixels for the absolute x,y
     * coordinates. When using this transform PixelType will always be active.
     * Setting a CCD coordinate which does not correspond to an active pixel
     * will result in an error. The {@link #getGlobalX()} and
     * {@link #getGlobalY()} methods can be used to obtain the global coordinate
     * corresponding to the x,y given.
     */
    class ActiveTransform implements CCDTransform {

        private SegmentInterface segment;
        private int serial;
        private int parallel;
        private int x;
        private int y;
        private int globalX;
        private int globalY;
        private final CCDInterface ccd;
        
        ActiveTransform(CCDInterface ccd) {
            this.ccd = ccd;
        }
        

        @Override
        public void setXY(int x, int y) {
            assert (x >= 0 && x < ccd.getActiveSerialSize());
            assert (y >= 0 && y < ccd.getActiveParallelSize());
            this.x = x;
            this.y = y;
            int column = x / ccd.getType().getSegmentSerialActiveSize();
            int row = y / ccd.getType().getSegmentParallelActiveSize();
            segment = ccd.getSegment(row,column);
            globalX = x + segment.getSerialPosition() * (segment.getSegmentSerialPrescanSize() + segment.getSegmentSerialOverscanSize()) + (!segment.isReadoutLeft() ? segment.getSegmentSerialPrescanSize() : segment.getSegmentSerialOverscanSize());
            globalY = y + segment.getParallelPosition() * segment.getSegmentParallelOverscanSize() + (segment.isReadoutDown() ? 0 : segment.getSegmentParallelOverscanSize());
            x %= segment.getSegmentSerialActiveSize();
            y %= segment.getSegmentParallelActiveSize();
            serial = segment.getSegmentSerialPrescanSize() + (!segment.isReadoutLeft() ? x : segment.getSegmentSerialActiveSize() - 1 - x);
            parallel = segment.isReadoutDown() ? y : segment.getSegmentParallelActiveSize() - 1 - y;
        }

        @Override
        public SegmentInterface getSegment() {
            return segment;
        }

        @Override
        public int getSerial() {
            return serial;
        }

        @Override
        public int getParallel() {
            return parallel;
        }

        @Override
        public PixelType getPixelType() {
            return PixelType.ACTIVE;
        }

        @Override
        public int getGlobalX() {
            return globalX;
        }

        @Override
        public int getGlobalY() {
            return globalY;
        }

        @Override
        public String toString() {
            return "ActiveTransform{" + "segment=" + segment + ", serial=" + serial + ", parallel=" + parallel + ", x=" + x + ", y=" + y + '}';
        }

        @Override
        public void setSegmentSerialParallel(SegmentInterface segment, int serial, int parallel) {
            assert (segment.getParentCCD() == ccd);
            assert (0 <= parallel && parallel < segment.getSegmentParallelActiveSize());
            assert (segment.getSegmentSerialPrescanSize() <= serial && serial < segment.getSegmentSerialPrescanSize() + segment.getSegmentSerialActiveSize());
            this.segment = segment;
            this.serial = serial;
            this.parallel = parallel;
            serial -= segment.getSegmentSerialPrescanSize();
            int yy = segment.isReadoutDown() ? parallel : segment.getSegmentParallelActiveSize() - 1 - parallel;
            int xx = !segment.isReadoutLeft() ? serial : segment.getSegmentSerialActiveSize() - 1 - serial;
            yy += segment.getParallelPosition()* segment.getSegmentParallelActiveSize();
            xx += segment.getSerialPosition()* segment.getSegmentSerialActiveSize();
            this.x = xx;
            this.y = yy;
            globalX = x + segment.getSerialPosition() * (segment.getSegmentSerialPrescanSize() + segment.getSegmentSerialOverscanSize()) + (!segment.isReadoutLeft() ? segment.getSegmentSerialPrescanSize() : segment.getSegmentSerialOverscanSize());
            globalY = y + segment.getParallelPosition() * segment.getSegmentParallelOverscanSize() + (segment.isReadoutDown() ? 0 : segment.getSegmentParallelOverscanSize());
        }

        @Override
        public int getX() {
            return x;
        }

        @Override
        public int getY() {
            return y;
        }
    }

    /**
     * A CCDTransform which considers all pixels for the absolute x,y
     * coordinates.
     */
    class GlobalTransform implements CCDTransform {

        private SegmentInterface segment;
        private int serial;
        private int parallel;
        private PixelType pixelType;
        private int x;
        private int y;
        private final CCDInterface ccd;
        
        GlobalTransform(CCDInterface ccd) {
            this.ccd = ccd;
        }

        @Override
        public void setXY(int x, int y) {
            assert (x >= 0 && x < ccd.getTotalSerialSize());
            assert (y >= 0 && y < ccd.getTotalParallelSize());
            this.x = x;
            this.y = y;
            int column = x / ccd.getType().getSegmentTotalSerialSize();
            int row = y / ccd.getType().getSegmentTotalParallelSize();
            segment = ccd.getSegment(row,column);
            y %= ccd.getType().getSegmentTotalParallelSize();
            x %= ccd.getType().getSegmentTotalSerialSize();
            parallel = segment.isReadoutDown() ? y : ccd.getType().getSegmentTotalParallelSize() - 1 - y;
            serial = segment.isReadoutLeft() ? x : ccd.getType().getSegmentTotalSerialSize() - 1 - x;
            if (parallel > segment.getSegmentParallelActiveSize()) {
                pixelType = PixelType.PARALLEL_OVERSCAN;
            } else if (serial < segment.getSegmentSerialPrescanSize()) {
                pixelType = PixelType.SERIAL_PRESCAN;
            } else if (serial > segment.getSegmentSerialPrescanSize() + segment.getSegmentSerialActiveSize()) {
                pixelType = PixelType.SERIAL_OVERSCAN;
            } else {
                pixelType = PixelType.ACTIVE;
            }
        }

        @Override
        public void setSegmentSerialParallel(SegmentInterface segment, int serial, int parallel) {
            assert (segment.getParentCCD() == ccd);
            assert (0 <= parallel && parallel < ccd.getType().getSegmentTotalParallelSize());
            assert (0 <= serial && serial < ccd.getType().getSegmentTotalSerialSize());
            this.segment = segment;
            this.serial = serial;
            this.parallel = parallel;
            if (parallel > segment.getSegmentParallelActiveSize()) {
                pixelType = PixelType.PARALLEL_OVERSCAN;
            } else if (serial < segment.getSegmentSerialPrescanSize()) {
                pixelType = PixelType.SERIAL_PRESCAN;
            } else if (serial > segment.getSegmentSerialPrescanSize() + segment.getSegmentSerialActiveSize()) {
                pixelType = PixelType.SERIAL_OVERSCAN;
            } else {
                pixelType = PixelType.ACTIVE;
            }
            int xx = segment.isReadoutLeft() ? serial : ccd.getType().getSegmentTotalSerialSize() - 1 - serial;
            int yy = segment.isReadoutDown() ? parallel : ccd.getType().getSegmentTotalParallelSize() - 1 - parallel;
            yy += segment.getParallelPosition()* ccd.getType().getSegmentTotalParallelSize();
            xx += segment.getSerialPosition()* ccd.getType().getSegmentTotalSerialSize();
            this.x = xx;
            this.y = yy;
        }

        @Override
        public SegmentInterface getSegment() {
            return segment;
        }

        @Override
        public int getSerial() {
            return serial;
        }

        @Override
        public int getParallel() {
            return parallel;
        }

        @Override
        public PixelType getPixelType() {
            return pixelType;
        }

        @Override
        public String toString() {
            return "GlobalTransform{" + "segment=" + segment + ", serial=" + serial + ", parallel=" + parallel + ", pixelType=" + pixelType + ", x=" + x + ", y=" + y + '}';
        }

        @Override
        public int getGlobalX() {
            return x;
        }

        @Override
        public int getGlobalY() {
            return y;
        }

        @Override
        public int getX() {
            return x;
        }

        @Override
        public int getY() {
            return y;
        }

    }
}
