
package org.lsst.ccs.utilities.ccd;

import java.util.List;

/**
 * An interface to describe CCDs.
 * 
 * @author turri
 */
public interface CCDInterface {
    
    /**
     * The active parallel size of this CCD. 
     * This takes into account only the exposed part of a CCD and does not
     * account for the over scan region.
     * 
     * @return The overall parallel active size of this CCD.
     */    
    public int getActiveParallelSize();
    
    /**
     * The active serial size of this CCD. 
     * This takes into account only the exposed part of a CCD and does not
     * account for the over and pre scan regions.
     * 
     * @return The overall serial active size of this CCD.
     */    
    public int getActiveSerialSize();
    
    /**
     * The total parallel size of this CCD. 
     * This takes into account the exposed size of the CCD and the over scan
     * regions.
     * 
     * @return The total parallel size of this CCD.
     */    
    public int getTotalParallelSize();
    
    /**
     * The total serial size of this CCD. 
     * This takes into account the exposed part of a CCD and the over and pre 
     * scan regions.
     * 
     * @return The total serial size of this CCD.
     */    
    public int getTotalSerialSize();
    
    /**
     * Get the number of segments in the Parallel direction
     * @return the number of segments in the parallel direction.
     */
    public int getParallelSegmentCount();
    
    /**
     * Get the number of segments in the Serial direction
     * @return the number of segments in the serial direction.
     */
    public int getSerialSegmentCount();

    /**
     * Get SegmentInterface at position (p,s).
     * 
     * @param p Position along the parallel direction
     * @param s Position along the serial direction
     * @return The SegmentInterface at the requested position.
     */
    public SegmentInterface getSegment(int p, int s);
    
    /**
     * The list of segments in readout order.
     *
     * @return An unmodifiable list of segments, with item 0 corresponding to
     * amplifier 1.
     */
    public List<SegmentInterface> getSegments();
    
    /**
     * Get the CCDTye for this CCD.
     * 
     * @return The CCDType
     */
    public CCDType getType();
    
    
    /**
     * The CCD dimension, in pixels, of the active area along the parallel direction.
     * This is the segments parallel size multiplied by the number of segments
     * laid out along the parallel direction.
     * @return The CCD dimension, in pixels, of the active area along the parallel direction.
     */
    public int getActiveAreaParallelSize();
    
    /**
     * The CCD dimension, in pixels, of the active area along the serial direction.
     * This is the segments serial size multiplied by the number of segments
     * laid out along the serial direction.
     * @return The CCD dimension, in pixels, of the active area along the serial direction.
     */
    public int getActiveAreaSerialSize();
    
    /**
     * The CCD dimension, in pixels, of the physical area along the parallel direction.
     * This is the sum of the segments parallel size multiplied by the number of segments
     * laid out along the parallel direction and twice the CCD edge on the parallel direction.
     * @return The CCD dimension, in pixels, of the physical area along the parallel direction.
     */
    public int getPhysicalAreaParallelSize();
    
    /**
     * The CCD dimension, in pixels, of the physical area along the serial direction.
     * This is the sum of the segments serial size multiplied by the number of segments
     * laid out along the serial direction and twice the CCD edge on the serial direction.
     * @return The CCD dimension, in pixels, of the physical area along the serial direction.
     */
    public int getPhysicalAreaSerialSize(); 
    
}
