package org.lsst.ccs.utilities.ccd;

import java.awt.Dimension;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.lsst.ccs.geometry.Geometry;

public class CCD extends Geometry<Segment> implements CCDInterface {

    private final CCDType ccdType;
    private final List<SegmentInterface> segments = new ArrayList<>();

    public CCD(String label, CCDType ccdType) {
        this(2, 8, ccdType, label);
    }

    /**
     * Create a CCDGeomtry object. To create geometry objects for standard CCD
     * types see the {@link CCDType} class.
     *
     * @param serialSegmentsCount The number of segments in the serial direction
     * @param parallelSegmentsCount The number of segments in the parallel direction
     * @param ccdType The type of this CCD.
     * @param label The CCD Label
     */
    CCD(int parallelSegmentsCount, int serialSegmentsCount, CCDType ccdType, String name) {
        super(name, new Dimension(ccdType.getCCDpx(), ccdType.getCCDpy()),parallelSegmentsCount, serialSegmentsCount);
        this.ccdType = ccdType;
    }

    /**
     * Add a Segment to this CCD at a given serial and parallel position.
     *
     * @param segment The Segment to add to the CCD.
     * @param s The Segment serial position.
     * @param p The Segment parallel position.
     */
    @Override
    protected void addGeometryToGrid(Segment segment, int p, int s) {
        int x = (int)ccdType.getParallelEdge() + (getParallelChildrenCount()-1-p)*segment.getWidth();
        int y = (int)ccdType.getSerialEdge()+(getSerialChildrenCount()-1-s) * segment.getHeight();
        addGeometry(segment, x, y);
        segments.add(segment);
        Collections.sort(segments, (SegmentInterface o1, SegmentInterface o2) -> o1.getChannel() > o2.getChannel() ? 1 : -1);

    }

    @Override
    public int getTotalSerialSize() {
        return ccdType.getSegmentTotalSerialSize() * getSerialChildrenCount();
    }

    @Override
    public int getActiveSerialSize() {
        return ccdType.getSegmentSerialActiveSize() * getSerialChildrenCount();
    }

    @Override
    public int getTotalParallelSize() {
        return ccdType.getSegmentTotalParallelSize() * getParallelChildrenCount();
    }

    @Override
    public int getActiveParallelSize() {
        return ccdType.getSegmentParallelActiveSize() * getParallelChildrenCount();
    }

    @Override
    public SegmentInterface getSegment(int p, int s) {
        return getChild(p, s);
    }

    @Override
    public CCDType getType() {
        return ccdType;
    }

    @Override
    public List<SegmentInterface> getSegments() {
        return Collections.unmodifiableList(segments);
    }

    @Override
    public int getParallelSegmentCount() {
        return getParallelChildrenCount();
    }

    @Override
    public int getSerialSegmentCount() {
        return getSerialChildrenCount();
    }

    @Override
    public int getActiveAreaParallelSize() {
        return ccdType.getActiveAreaParallelSize();
    }
    
    @Override
    public int getActiveAreaSerialSize() {
        return ccdType.getActiveAreaSerialSize();
    }
    
    @Override
    public int getPhysicalAreaParallelSize() {
        return ccdType.getPhysicalAreaParallelSize();
    }
    
    @Override
    public int getPhysicalAreaSerialSize() {
        return ccdType.getPhysicalAreaSerialSize();
    }
    
    public static CCD createCCD(String label, CCDType ccdType) {

        CCD ccd = new CCD(label, ccdType);
        for (int s = 0; s < ccd.getSerialChildrenCount(); s++) {
            for (int p = 0; p < ccd.getParallelChildrenCount(); p++) {
                int channel = p == 0 ? 16 - s : 1 + s;
                Segment segment = Segment.createCCDSegment(ccdType, "Seg" + p + s, s, p, channel);
                ccd.addChildGeometry(segment, p, s);
            }
        }
        return ccd;
    }

}
