package org.lsst.ccs.geometry;

import java.awt.Dimension;
import java.awt.Point;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Base Geometry: a rectangle inside which it is possible to add other
 * Geometries.
 *
 * Note on coordinate system and dimensions.
 * The (x,y) coordinate system refers to Java Image/Component
 * convention. (0,0) is the upper left corner with x increasing to the right
 * and y increasing downward. For simplicity (x,y) cannot be negative.
 * The "width" of a Geometry is its maximum extent along the x direction and the
 * "height" is along the y direction.
 * 
 * In the Camera Coordinate System (CCS) the coordinates are defined as follows: 
 *  - (x,y) The x direction grows to the left, the y upward. In this coordinate system
 * the focal plan is as viewed from L3, i.e. looking down at the CCDs
 *  - (p,s) The parallel, serial coordinate system is aligned along the (x,y) Camera
 * Coordinate System: the parallel direction grows along x and the serial direction 
 * grows along the y direction.
 *
 * A Geometry upper left corner is ALWAYS (0,0). This point will be always used
 * when evaluating dimensions.
 *
 * In the Camera Coordinate System the bottom right corner of Segment00, CCD00, Raft00
 * is defined as (1,1). The pixels grow along the (x,y) Camera Coordinate System
 * 
 * This class is not thread safe.
 *
 * @author turri
 * @param <T> The template of the classes that can be added to this Geometry
 */
public class Geometry<T extends Geometry> {

    private final Dimension dimension;
    private final List<InnerGeometry<T>> components = new ArrayList<>();
    private Geometry parent;
    private boolean hasFixedDimension = false;
    private final String name;
    private final Geometry[][] children; 
    protected final int serialChildrenCount, parallelChildrenCount;
    protected int serialPosition, parallelPosition;

    /**
     * Create a new Geometry with a maximum set of components in the (p,s) 
     * coordinate system.
     * 
     * Subclasses of Geometries that allow children to be added must override
     * method {@link #addGeometryToGrid(org.lsst.ccs.geometry.Geometry, int, int) addGeometryToGrid}.
     * 
     * @param name The name of this Geometry
     * @param dimension The dimension of this Geometry. Children must fit within this dimension.
     * @param p The number of allowed components in the parallel direction.
     * @param s The number of allowed components in the serial direction
     */
    public Geometry(String name, Dimension dimension, int p, int s) {
        this.dimension = dimension;
        this.hasFixedDimension = true;
        this.name = name;
        children = ( p > 0 && s > 0 ) ? new Geometry[p][s] : null;
        this.serialChildrenCount = s;
        this.parallelChildrenCount = p;
    }

    /**
     * Create a new Geometry with no children.
     * If an attempt is made to add children an exception will be thrown.
     * 
     * @param name The name of this Geometry
     * @param dimension The dimension of this Geometry.
     */
    public Geometry(String name, Dimension dimension) {
        this(name, dimension, 0, 0);
    }
    
    /**
     * Get the Geometry name
     * @return The Geometry name.
     */
    public String getName() {
        return name;
    }
    
    /**
     * Get the Geometry width.
     *
     * @return The width of the Geometry;
     */
    public int getWidth() {
        return dimension.width;
    }

    /**
     * Get the Geometry height;
     *
     * @return The height of the Geometry;
     */
    public int getHeight() {
        return dimension.height;
    }

    /**
     * Get the Dimension of this Geometry.
     *
     * @return The Dimension of this Geometry.
     */
    public Dimension getDimension() {
        return new Dimension(dimension);
    }

    /**
     * Get the parent of this Geometry. Null will be returned if this is an
     * orphan.
     *
     * @return The parent (null if this is an orphan).
     */
    public Geometry getParent() {
        return parent;
    }

    /**
     * Check if this Geometry has a Parent.
     *
     * @return true if this Geometry has a parent.
     *
     */
    public boolean hasParent() {
        return getParent() != null;
    }

    protected final void setParent(Geometry parent) {
        if (this.parent != null) {
            throw new RuntimeException("This Geometry already already has a parent!");
        }
        this.parent = parent;
    }

    /**
     * Add a Geometry to this Geometry at a given (x,y) location.
     * The convention is the the new Geometry upper left corner will be placed
     * at (x,y).
     *
     * Child Geometries cannot be added to a Geometry that already has a parent AND
     * does not have a fixed dimension. The reason for this choice is to simplify
     * the logic of boundaries breaking.
     *
     * Coordinate x and y must be positive otherwise an exception will be thrown.
     *
     * If the this parent Geometry has fixed boundaries an exception will be thrown if the
     * child Geometry to be added does not fit inside its boundaries.
     *
     * An exception will be thrown if the child Geometry being added already
     * belongs to another Geometry.
     *
     * @param child The Geometry to be added
     * @param x The x coordinate of the child Geometry location
     * @param y The y coordinate of the child Geometry location
     */
    protected void addGeometry(T child, int x, int y) {
        if (getParent() != null && !hasFixedDimension) {
            throw new RuntimeException("Cannot add a child Geometry to a non orfan parent Geometry"
                    + " with variable dimensions");
        }
        if (child.getParent() != null) {
            throw new RuntimeException("The child geometry already belongs to another Geometry.");
        }
        if (x < 0 || y < 0) {
            throw new IllegalArgumentException("Illegal placement position (" + x + "," + y + "), coordiantes cannot be negative");
        }
        if (hasFixedDimension) {
            if (x > getWidth() || y > getHeight()) {
                throw new IllegalArgumentException("Illegal placement position (" + x + "," + y + "), it is outside of this geometry's size " + getWidth() + "x" + getHeight());
            }
            if (x + child.getWidth() > getWidth() || y + child.getHeight() > getHeight()) {
                throw new IllegalArgumentException("The child Geometry cannot be added within the boundaries of this Geometry " + getWidth() + "x" + getHeight());
            }
        }

        child.setParent(this);
        Point point = new Point(x, y);
        components.add(new InnerGeometry<>(child, point));
        if (!hasFixedDimension) {
            dimension.setSize(Math.max(getWidth(), x + child.getWidth()), Math.max(getHeight(), y + child.getHeight()));
        }
    }


    public int getSerialChildrenCount() {
        return this.serialChildrenCount;
    }

    public int getParallelChildrenCount() {
        return this.parallelChildrenCount;
    }
    
    public int getSerialPosition() {
        return this.serialPosition;
    }

    public int getParallelPosition() {
        return this.parallelPosition;
    }
    
    public void setSerialPosition(int serialPosition) {
        this.serialPosition = serialPosition;
    }

    public void setParallelPosition(int parallelPosition) {
        this.parallelPosition = parallelPosition;
    }
    
    /**
     * Add a child to the grid to a given position.
     * @param child The child to be added
     * @param s     The serial position to add it to 
     * @param p     The parallel position to add it to
     */    
    public void addChildGeometry(T child, int p, int s) {
        if ( children == null ) {
            throw new RuntimeException("This geometry does not allow children.");
        }
        children[p][s] = child;
        child.parallelPosition = p;
        child.serialPosition = s;
        addGeometryToGrid(child, p, s);
    }
    
    protected void addGeometryToGrid(T child, int p, int s) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }
    
    /**
     * Get child at position (p,s).
     * @param p     The parallel position to add it to
     * @param s     The serial position to add it to 
     * @return      The child at the given position.
     * 
     */
    public T getChild(int p, int s) {
        return (T)children[p][s];
    }
    
    /**
     * Check if this Geometry has children.
     *
     * @return true if this Geometry has children.
     */
    public boolean hasChildren() {
        return !components.isEmpty();
    }

    /**
     * Get the children for this Geometry.
     * It returns a map of Geometry, Point where the Point is the location
     * of the child Geometry in the topmost geometry.
     *
     * @return A map of Geometry, Point
     */
    public final Map<Geometry, Point> getChildrenWithAbsoluteCoordinates() {
        HashMap<Geometry, Point> map = new HashMap<>();
        for (InnerGeometry ig : components) {
            map.put(ig.getGeometry(), getAbsolutePoint(ig.getPoint()));
        }
        return map;
    }

    /**
     * Get the list of children for this Geometry.
     *
     * @return A list of children Geometry objects
     */
    public final List<T> getChildrenList() {
        ArrayList<T> c = new ArrayList<>();
        for (InnerGeometry<T> ig : components) {
            c.add(ig.getGeometry());
        }
        return c;
    }

    /**
     * Get this Geometry unique id.
     * This is built by going over the parents and concatenating their name
     * separated by columns.
     * @return The Geometry unique id.
     * 
     */
    public final String getUniqueId() {
        return buildUniqueId(this,"");        
    }
    
    private static String buildUniqueId(Geometry g, String name) {
        name = name.isEmpty() ? g.getName() : g.getName()+"."+name;
        if ( g.hasParent() ) {
            return buildUniqueId(g.getParent(), name);
        }
        return name;
    }
    
    /**
     * Find a Geometry by name.
     * 
     * @param geometryUniqueId The unique id of the geometry to be found.
     * @return The geometry or null if the geometry is not found
     * 
     */
    public final Geometry findGeometry(String geometryUniqueId) {
        return findGeometry(this, geometryUniqueId);        
    }
    
    private static Geometry findGeometry(Geometry<?> g, String geometryUniqueId) {
        if (g.getUniqueId().equals(geometryUniqueId)) {
            return g;
        }
        for ( Geometry child : g.getChildrenList() ) {
            Geometry found = findGeometry(child, geometryUniqueId);
            if ( found != null ) {
                return found;
            }
        }
        return null;
    }
    /**
     * Get the number of children in this Geometry.
     *
     * @return the number of children in the geometry.
     */
    public final int getNumberOfChildren() {
        return components.size();
    }

    public Point getGeometryAbsolutePosition() {
        return getAbsolutePoint(new Point(0,0));
    }
    
    public Point getAbsolutePoint(Point p) {
        if (!hasParent()) {
            return p;
        }
        Point thisGeometryPoint = getParent().getGeometryPoint(this);
        Point p1 = new Point(p.x+thisGeometryPoint.x, p.y+thisGeometryPoint.y);
        return getParent().getAbsolutePoint(p1);
    }

    /**
     * Get the location Point of a child Geometry within this Geometry.
     *
     * @param geometry The child Geometry.
     * @return The Point where the provided GEometry is attached to this Geometry.
     * 
     */
    public Point getGeometryPoint(Geometry geometry) {
        for (InnerGeometry ig : components) {
            if (ig.getGeometry() == geometry) {
                return ig.getPoint();
            }
        }
        throw new RuntimeException("Could not find location for this Geometry: " + geometry);
    }

    private class InnerGeometry<K extends Geometry> {

        private final K geometry;
        private final Point point;

        InnerGeometry(K geometry, Point p) {
            this.geometry = geometry;
            this.point = p;
        }

        K getGeometry() {
            return geometry;
        }

        int getXPosition() {
            return point.x;
        }

        int getYPosition() {
            return point.y;
        }

        Point getPoint() {
            return point;
        }

    }

}
