package org.lsst.ccs.utilities.image;

import java.io.IOException;
import java.io.InputStream;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.bootstrap.BootstrapResourceUtils;

/**
 * Utility class for the generation of header specifications objects.
 * These objects create a map between fits header keywords and quantities that
 * are loaded at runtime.
 * Each spec file is linked to a named header. Default name headers that must be
 * provided when writing fits files are "primary" for the primary header and
 * "extended" for any data header file extension, i.e. extensions containing
 * image data.
 * Any additional header name (beside "primary" and "extended") will create an
 * additional extension in which the fits keywords are stored in a binary table.
 *
 * @author turri
 */
public class FitsHeadersSpecifications {

    
    private static final Map<String, HeaderSpecification> imageWriterConfig = new HashMap<>();

    
    /**
     * Add the a header specification file. This information will be used to 
     * generate the Header configuration map. The name of the file (excluding 
     * the .spec extension if provided) will be used as
     * the name of the header extension.
     * @param fileName The name of the file to be loaded. The extension may be omitted.
     *                 No path information is to be provided. The file will be loaded
     *                 as a Bootstrap resource.
     */
    public static final void addSpecFile(String fileName) {
        readHeaderSpecification(fileName, null);
    }

    /**
     * Add the a header specification file. This information will be used to 
     * generate the Header configuration map. The name of the file (excluding 
     * the .spec extension if provided) will be used as
     * the name of the header extension unless a corresponding header name is provided.
     * @param fileName The name of the file to be loaded. The extension may be omitted.
     *                 No path information is to be provided. The file will be loaded
     *                 as a Bootstrap resource.
     * @param headerName The headers name for which this spec file is added
     *                   If no header name is provided, the file name will be used.
     * 
     */
    public static final void addSpecFile(String fileName, String headerName) {
        readHeaderSpecification(fileName, headerName);
    }
        
    /**
     * Get the HeaderSpecifications as built from the read spec files.
     * @return The Map of HeaderSpecifications;
     */    
    public static final Map<String, HeaderSpecification> getHeaderSpecifications() {
        return Collections.unmodifiableMap(imageWriterConfig);
    }

    /**
     * Remove Header specification. Remove the Header specification for the 
     * provided header name.
     * @param headerName
     */
    public final void removeHeaderSpecification(String headerName) {
        imageWriterConfig.remove(headerName);
    }
    
    /**
     * Remove all Header specifications. Remove all the Header specifications 
     * currently loaded.
     */
    public static final void removeAllHeaderSpecifications() {
        imageWriterConfig.clear();
    }
    
    
    private static void readHeaderSpecification(String name, String headerName) throws RuntimeException {
        final String fileName = name.endsWith(".spec") ? name : name + ".spec";
        if ( headerName == null ) {
            headerName = fileName.replace(".spec", "");
        }
        if ( imageWriterConfig.containsKey(headerName) ) {
            System.out.println("WARNING : Replacing existing header specification "+name);
//            throw new IllegalArgumentException("Header name "+headerName+" already exists and cannot be overwritten.");
        }
        try (InputStream bootstrapResource = BootstrapResourceUtils
                .getBootstrapResource(fileName)) {
            if (bootstrapResource == null) {
                System.out.println("Failed to open spect file "+fileName);
                return;
            }
            HeaderSpecification desc = new HeaderSpecification(headerName,
                    bootstrapResource);
            imageWriterConfig.put(headerName, desc);
        } catch (IOException x) {
            throw new RuntimeException("Error reading " + fileName, x);
        }
    }
    
}
