package org.lsst.ccs.utilities.image;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import nom.tam.fits.BasicHDU;
import nom.tam.fits.FitsException;
import nom.tam.fits.FitsFactory;
import nom.tam.fits.FitsUtil;
import nom.tam.fits.Header;
import nom.tam.fits.HeaderCardException;
import nom.tam.util.BufferedFile;
import org.lsst.ccs.utilities.ccd.image.data.RawImageData.BitsPerPixel;
import static org.lsst.ccs.utilities.image.HeaderSpecification.DataType.Float;
import org.lsst.ccs.utilities.image.HeaderSpecification.HeaderLine;
import org.lsst.ccs.utilities.image.ImageSet.Image;

/**
 * A utility for writing FITS files following LSST conventions.
 *
 * @author tonyj
 */
public class FitsFileWriter implements Closeable {

    //A static object to be used when writing fits images.
    //It generates objects that provide metadata information needed to fill the headers
    private static FitsHeaderMetadataProviderFactory metadataProviderFactory = new DefaultFitsHeaderMetadataProviderFactory();
        
    private class ImageExtension {

        private final long startPosition;
        private final long endPosition;
        private long currentPosition;
        private final FitsCheckSum.Checksum checkSum;
        private final Header header;

        private ImageExtension(long start, long current, long end, Header header) {
            this.startPosition = start;
            this.currentPosition = current;
            this.endPosition = end;
            this.header = header;
            this.checkSum = new FitsCheckSum.Checksum();
        }

        private void write(ByteBuffer src) throws IOException {
            int length = src.remaining();
            if (length + currentPosition > endPosition) {
                throw new IOException("Too much data written for image "+length+" "+currentPosition+" "+endPosition);
            }
            FitsCheckSum.updateChecksum(src, checkSum);
            bf.seek(currentPosition);
            if (src.hasArray()) {
                bf.write(src.array(), src.arrayOffset() + src.position(), src.remaining());
                src.position(src.limit());
            } else {
                while (src.remaining() > 0) {
                    bf.write(src.get());
                }
            }
            currentPosition += length;
        }

        private void updateDataSum() throws IOException {
            try {
                FitsCheckSum.updateDataSum(header, checkSum.getCheckSum());
                bf.seek(startPosition);
                header.write(bf);
            } catch (FitsException ex) {
                throw new IOException("Unable to add datasum to header", ex);
            }
        }
    }
    private BufferedFile bf;
    private ImageExtension[] imageExtensions;

    /**
     * Open an LSST FITS file for writing a CCD ImageSet.
     *
     * @param file The file to write to
     * @param images The ImageSet to write to the file. Note that this specifies
     * the images to write, not the actual data for the images.
     * @param metaData The meta-data maps to use to extract header info from
     * @param config The configuration which controls how meta data is written
     * to the file
     * @param bits The number of bits per pixel for images
     * @throws java.io.IOException
     * @throws nom.tam.fits.FitsException
     */
    @Deprecated
    public FitsFileWriter(File file, ImageSet images, Map<String, Map<String, Object>> metaData, Map<String, HeaderSpecification> config, BitsPerPixel bits) throws IOException, FitsException {
        initializeFitsFileWriter(file, images, metaData, config, bits);
    }

    private void initializeFitsFileWriter(File file, ImageSet images, Map<String, Map<String, Object>> metaData, Map<String,HeaderSpecification> config, BitsPerPixel bits) throws IOException, FitsException {
        if ( metadataProviderFactory == null ) {
            throw new RuntimeException("Cannot invoke this constructor if a FitsHeaderMetadataProviderFactory is not registered with this FitsFileWriter object.");
        }
        FitsHeaderMetadataProvider provider = getFitsHeaderMetadataProviderFactory().createFitsHeaderMetadataProvider(images);
        if ( provider == null ) {
            throw new RuntimeException("Could not create a FitsHeaderMetadataProvider object for ImageSet: "+images);
        }

        imageExtensions = new ImageExtension[images.getImages().size()];
        int[][] intDummyData = new int[1][1];
        short[][] shortDummyData = new short[1][1];
        Object[] tableDummyData = new Object[0];
        Object dummyData = bits == BitsPerPixel.BIT16 ? shortDummyData : intDummyData;
        bf = new BufferedFile(file, "rw");

        FitsFactory.setUseHierarch(true);
        
        Map<String, Map<String, Object>> fullMetaData = provider.getPrimaryHeaderMetadata();
        Map<String,Object> imageSetMetadata = images.getMetaData();
        if ( imageSetMetadata != null ) {
            fullMetaData.put("imageSet", imageSetMetadata);
        }
        
        if ( metaData != null ) {
            for ( String k : metaData.keySet() ) {
                fullMetaData.put(k, metaData.get(k));
            }
        }
        
        
        // Create primary header
        BasicHDU primary = BasicHDU.getDummyHDU();
        addMetaDataToHeader(primary, "primary", fullMetaData, config);
        FitsCheckSum.setChecksum(primary);
        primary.getHeader().write(bf);

        //Create extension image headers for each image, and reserve space for data
        int i = 0;
        for (Image image : images.getImages()) {
            BasicHDU hdu = FitsFactory.HDUFactory(dummyData);
            
            Map<String,Map<String,Object>> extendedMetadata = provider.getDataExtendedHeaderMetadata(i);
            Map<String, Map<String,Object>> imageMetadata = images.getImages().get(i).getMetaData();
            if ( imageMetadata != null ) {
                extendedMetadata.putAll(imageMetadata);
            }
            addMetaDataToHeader(hdu, "extended", extendedMetadata, config);
            if (bits == BitsPerPixel.BIT16) {
                // To store as unsigned 16 bit values, we have to set BZERO to 32768, and 
                // subtract 32768 from each value. 
                // See: http://heasarc.gsfc.nasa.gov/docs/software/fitsio/c/c_user/node23.html
                hdu.addValue("BSCALE", 1.0, "Unsigned 16 bit data");
                hdu.addValue("BZERO", 32768, "Unsigned 16 bit data");
            }
            Header header = hdu.getHeader();
            header.setXtension("IMAGE");
            header.setNaxis(1, image.getWidth());
            header.setNaxis(2, image.getHeight());
            FitsCheckSum.setChecksum(hdu);
            long start = bf.getFilePointer();
            header.write(bf);
            long current = bf.getFilePointer();
            long imageSize = bits.bytes() * image.getWidth() * image.getHeight();
            bf.seek(bf.getFilePointer() + imageSize);
            FitsUtil.pad(bf, imageSize);
            imageExtensions[i++] = new ImageExtension(start, current, bf.getFilePointer(), header);
        }

        // If necessary, create any additional extended HDU's here.
        // Create any extra BinTables from the specification
        FitsFactory.setUseAsciiTables(false);
        for (String key : config.keySet()) {
            if (!"primary".equals(key) && !"extended".equals(key)) {
                BasicHDU binary = FitsFactory.HDUFactory(tableDummyData);

                Map<String, Map<String, Object>> additionalMetaData = provider.getAdditionalExtendedHeaderMetadata(key);
                
                addMetaDataToHeader(binary, key, additionalMetaData, config);
                Header header = binary.getHeader();
                header.setXtension("BINTABLE");
                FitsCheckSum.setChecksum(binary);
                header.write(bf);
            }
        }
    }
    
    /**
     * Construct a FitsFileWriter object for the provided ImageSet.
     * By default this will write 32 bit images.
     * 
     * @param file The name of the output fits file
     * @param images The ImageSet of images to write in the Data extended headers.
     * @throws IOException
     * @throws FitsException 
     */
    public FitsFileWriter(File file, ImageSet images) throws IOException, FitsException {
        this(file, images, null, BitsPerPixel.BIT32);
    }
    
    public FitsFileWriter(File file, ImageSet images, Map<String, Map<String, Object>> metaData) throws IOException, FitsException {
        this(file, images, metaData, BitsPerPixel.BIT32);
    }
    public FitsFileWriter(File file, ImageSet images, Map<String, Map<String, Object>> metaData, BitsPerPixel bits) throws IOException, FitsException {
        initializeFitsFileWriter(file, images, metaData, FitsHeadersSpecifications.getHeaderSpecifications(), bits);
    }
        

    /**
     * Write the actual image data to the file. It is not necessary that all of
     * the data for the image be available at once, this method will write
     * whatever data is currently available in the byte buffer to the specified
     * image, and will keep track of how much has been written to each image to
     * allow more data to be written later. This method assumes the data is
     * given in the order it is to be written to the file. If any data
     * reordering is needed it needs to be done before calling this method.
     *
     * @param imageIndex The image to which this data is to be written
     * @param src The image
     * @throws IOException If an IOException is generated, or if more data is
     * sent than was expected for a particular image.
     */
    public void write(int imageIndex, ByteBuffer src) throws IOException {
        imageExtensions[imageIndex].write(src);
    }

    @Override
    public void close() throws IOException {
        // Update the datasum keywords in place
        for (ImageExtension imageExtension : imageExtensions) {
            imageExtension.updateDataSum();
        }
        bf.close();
    }

    private void addMetaDataToHeader(BasicHDU hdu, String specName, Map<String, Map<String, Object>> metaData, Map<String, HeaderSpecification> config) throws HeaderCardException, IOException {
        HeaderSpecification spec = config.get(specName);        
        if (spec == null) {
            throw new IOException("Missing specification for header: " + specName);
        }
        for (HeaderLine header : spec.getHeaders()) {
            Object value = header.getValue(metaData);
            try {
                if (value != null) {
                    switch (header.getDataType()) {
                        case Integer:
                            hdu.addValue(header.getKeyword(), ((Number) value).intValue(), header.getComment());
                            break;
                        case Float:
                            double data = ((Number) value).doubleValue();
                            if (!Double.isFinite(data)) {
                                throw new IllegalArgumentException("Can not store non-finite floating point in FITS file");
                            }
                            hdu.addValue(header.getKeyword(), data, header.getComment());
                            break;
                        case Boolean:
                            hdu.addValue(header.getKeyword(), (Boolean) value, header.getComment());
                            break;
                        case Date:
                            hdu.addValue(header.getKeyword(), DateUtils.convertDateToString((Date) value), header.getComment());
                            break;
                        case MJD:
                            hdu.addValue(header.getKeyword(), DateUtils.convertDateToMJD((Date) value), header.getComment());
                            break;
                        default:
                            hdu.addValue(header.getKeyword(), String.valueOf(value), header.getComment());
                    }
                }
            } catch (ClassCastException x) {
                throw new IOException(String.format("Meta-data header %s with value %s(%s) cannot be converted to type %s", header.getKeyword(), value, value.getClass(), header.getDataType()));
            }
        }
    }
    
    public static void setFitsHeaderMetadataProviderFactory(FitsHeaderMetadataProviderFactory metadataProviderFactory) {        
        FitsFileWriter.metadataProviderFactory = metadataProviderFactory;
    }

    public static FitsHeaderMetadataProviderFactory getFitsHeaderMetadataProviderFactory() {        
        return metadataProviderFactory;
    }
    
}
