package org.lsst.ccs.geometry;

import java.awt.Dimension;
import java.awt.Point;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Base Geometry: a rectangle inside which it is possible to add other
 * Geometries.
 *
 * Note on coordinate system and dimensions.
 * The (x,y) coordinate system refers to Java Image/Component
 * convention. (0,0) is the upper left corner with x increasing to the right
 * and y increasing downward. For simplicity (x,y) cannot be negative.
 * The "width" of a Geometry is its maximum extent along the x direction and the
 * "height" is along the y direction.
 *
 * A Geometry upper left corner is ALWAYS (0,0). This point will be always used
 * when evaluating dimensions.
 *
 * This class is not thread safe.
 *
 * @author turri
 * @param <T> The template of the classes that can be added to this Geometry
 */
public class Geometry<T extends Geometry> {

    private Dimension dimension;
    private final List<InnerGeometry<T>> components = new ArrayList<>();
    private Geometry parent;
    private boolean hasFixedDimension = false;
    private final String name;
    private Geometry[][] children; 
    protected final int serialChildrenCount, parallelChildrenCount;
    protected int serialPosition, parallelPosition;

    /**
     * Create a new Geometry given a width and a height.
     * This geometry has a fixed size and it cannot be changed.
     * Geometries must be added within its boundaries.
     *
     * @param s The number of allowed components in the serial direction
     * @param p The number of allowed components in the parallel direction.
     * @param name The name of this Geometry
     */
    public Geometry(String name, int s, int p) {
        dimension = new Dimension(0, 0);
        this.name = name;
        children = new Geometry[s][p];
        this.serialChildrenCount = s;
        this.parallelChildrenCount = p;
    }

    /**
     * Create a new Geometry with no boundaries.
     * The dimension will be updated every time other Geometries are added.
     * @param name The name of this Geometry
     */
    public Geometry(String name) {
        dimension = new Dimension(0, 0);
        this.name = name;
        this.serialChildrenCount = 0;
        this.parallelChildrenCount = 0;
    }

    
    /**
     * Set a fixed dimension for this Geometry.
     * @param width The width of this Geometry
     * @param height The height of this Geometry
     * 
     */
    protected void setDimension(int width, int height) {
        this.dimension = new Dimension(width, height);
        hasFixedDimension = true;
    }
    
    
    /**
     * Get the Geometry name
     * @return The Geometry name.
     */
    public String getName() {
        return name;
    }
    
    /**
     * Get the Geometry width.
     *
     * @return The width of the Geometry;
     */
    public int getWidth() {
        return dimension.width;
    }

    /**
     * Get the Geometry height;
     *
     * @return The height of the Geometry;
     */
    public int getHeight() {
        return dimension.height;
    }

    /**
     * Get the Dimension of this Geometry.
     *
     * @return The Dimension of this Geometry.
     */
    public Dimension getDimension() {
        return new Dimension(dimension);
    }

    /**
     * Get the parent of this Geometry. Null will be returned if this is an
     * orphan.
     *
     * @return The parent (null if this is an orphan).
     */
    public Geometry getParent() {
        return parent;
    }

    /**
     * Check if this Geometry has a Parent.
     *
     * @return true if this Geometry has a parent.
     *
     */
    public boolean hasParent() {
        return getParent() != null;
    }

    protected final void setParent(Geometry parent) {
        if (this.parent != null) {
            throw new RuntimeException("This Geometry already already has a parent!");
        }
        this.parent = parent;
    }

    /**
     * Add a Geometry to this Geometry at a given (x,y) location.
     * The convention is the the new Geometry upper left corner will be placed
     * at (x,y).
     *
     * Child Geometries cannot be added to a Geometry that already has a parent AND
     * does not have a fixed dimension. The reason for this choice is to simplify
     * the logic of boundaries breaking.
     *
     * Coordinate x and y must be positive otherwise an exception will be thrown.
     *
     * If the this parent Geometry has fixed boundaries an exception will be thrown if the
     * child Geometry to be added does not fit inside its boundaries.
     *
     * An exception will be thrown if the child Geometry being added already
     * belongs to another Geometry.
     *
     * @param child The Geometry to be added
     * @param x The x coordinate of the child Geometry location
     * @param y The y coordinate of the child Geometry location
     */
    protected void addGeometry(T child, int x, int y) {
        if (getParent() != null && !hasFixedDimension) {
            throw new RuntimeException("Cannot add a child Geometry to a non orfan parent Geometry"
                    + " with variable dimensions");
        }
        if (child.getParent() != null) {
            throw new RuntimeException("The child geometry already belongs to another Geometry.");
        }
        if (x < 0 || y < 0) {
            throw new IllegalArgumentException("Illegal placement position (" + x + "," + y + "), coordiantes cannot be negative");
        }
        if (hasFixedDimension) {
            if (x > getWidth() || y > getHeight()) {
                throw new IllegalArgumentException("Illegal placement position (" + x + "," + y + "), it is outside of this geometry's size " + getWidth() + "x" + getHeight());
            }
            if (x + child.getWidth() > getWidth() || y + child.getHeight() > getHeight()) {
                throw new IllegalArgumentException("The child Geometry cannot be added within the boundaries of this Geometry " + getWidth() + "x" + getHeight());
            }
        }

        child.setParent(this);
        Point point = new Point(x, y);
        components.add(new InnerGeometry<>(child, point));
        if (!hasFixedDimension) {
            dimension.setSize(Math.max(getWidth(), x + child.getWidth()), Math.max(getHeight(), y + child.getHeight()));
        }
    }


    public int getSerialChildrenCount() {
        return this.serialChildrenCount;
    }

    public int getParallelChildrenCount() {
        return this.parallelChildrenCount;
    }
    
    public int getSerialPosition() {
        return this.serialPosition;
    }

    public int getParallelPosition() {
        return this.parallelPosition;
    }
    
    
    /**
     * Add a child to the grid to a given position.
     * @param child The child to be added
     * @param s     The serial position to add it to 
     * @param p     The parallel position to add it to
     */
    public void addChildGeometry(T child, int s, int p) {
        children[s][p] = child;
        child.parallelPosition = p;
        child.serialPosition = s;
        addGeometryToGrid(child, s, p);
    }
    
    protected void addGeometryToGrid(T child, int s, int p) {
        throw new UnsupportedOperationException("Not supported yet."); //To change body of generated methods, choose Tools | Templates.
    }
    
    /**
     * Get child at position (s,p).
     * @param s     The serial position to add it to 
     * @param p     The parallel position to add it to
     * @return      The child at the given position.
     * 
     */
    
    public T getChild(int s, int p) {
        return (T)children[s][p];
    }
    
    /**
     * Check if this Geometry has children.
     *
     * @return true if this Geometry has children.
     */
    public boolean hasChildren() {
        return !components.isEmpty();
    }

    /**
     * Get the children for this Geometry.
     * It returns a map of Geometry, Point where the Point is the location
     * of the child Geometry in the topmost geometry.
     *
     * @return A map of Geometry, Point
     */
    public final Map<Geometry, Point> getChildrenWithAbsoluteCoordinates() {
        HashMap<Geometry, Point> map = new HashMap<>();
        for (InnerGeometry ig : components) {
            map.put(ig.getGeometry(), getAbsolutePoint(ig.getPoint()));
        }
        return map;
    }

    /**
     * Get the list of children for this Geometry.
     *
     * @return A list of children Geometry objects
     */
    public final List<T> getChildrenList() {
        ArrayList<T> c = new ArrayList<>();
        for (InnerGeometry<T> ig : components) {
            c.add(ig.getGeometry());
        }
        return c;
    }

    /**
     * Get this Geometry unique id.
     * This is built by going over the parents and concatenating their name
     * separated by columns.
     * @return The Geometry unique id.
     * 
     */
    public final String getUniqueId() {
        return buildUniqueId(this,"");        
    }
    
    private static String buildUniqueId(Geometry g, String name) {
        name = name.isEmpty() ? g.getName() : g.getName()+"."+name;
        if ( g.hasParent() ) {
            return buildUniqueId(g.getParent(), name);
        }
        return name;
    }
    
    /**
     * Find a Geometry by name.
     * 
     * @param geometryUniqueId The unique id of the geometry to be found.
     * @return The geometry or null if the geometry is not found
     * 
     */
    public final Geometry findGeometry(String geometryUniqueId) {
        return findGeometry(this, geometryUniqueId);        
    }
    
    private static Geometry findGeometry(Geometry<?> g, String geometryUniqueId) {
        if (g.getUniqueId().equals(geometryUniqueId)) {
            return g;
        }
        for ( Geometry child : g.getChildrenList() ) {
            Geometry found = findGeometry(child, geometryUniqueId);
            if ( found != null ) {
                return found;
            }
        }
        return null;
    }
    /**
     * Get the number of children in this Geometry.
     *
     * @return the number of children in the geometry.
     */
    public final int getNumberOfChildren() {
        return components.size();
    }

    public Point getGeometryAbsolutePosition() {
        return getAbsolutePoint(new Point(0,0));
    }
    
    public Point getAbsolutePoint(Point p) {
        if (!hasParent()) {
            return p;
        }
        Point thisGeometryPoint = getParent().getGeometryPoint(this);
        Point p1 = new Point(p.x+thisGeometryPoint.x, p.y+thisGeometryPoint.y);
        return getParent().getAbsolutePoint(p1);
    }

    /**
     * Get the location Point of a component within its parent.
     *
     */
    public Point getGeometryPoint(Geometry geometry) {
        for (InnerGeometry ig : components) {
            if (ig.getGeometry() == geometry) {
                return ig.getPoint();
            }
        }
        throw new RuntimeException("Could not find location for this Geometry: " + geometry);
    }

    protected void setExpandedView(boolean expandedView) {
        
    }
    
    
    private void reset() {
        this.parent = null;
        this.components.clear(); 
        if ( ! hasFixedDimension ) {
            this.dimension = new Dimension(0,0);
        }
    }
    /**
     * Convert this geometry to its expanded version. This takes into account
     * the electronic view of segments (overscans and understans).
     */
    public static void convertToExpandedView(Geometry<?> geometry, boolean expandedView) {
        if ( geometry.getParent() != null ) {
            throw new IllegalArgumentException("Cannot convert a Geometry that belongs to a parent. The conversion must happen at the top level");
        }
        internalConvert(geometry, expandedView);
    }
    private static void internalConvert(Geometry geometry, boolean expandedView) {
        geometry.reset();
        for ( int s = 0; s < geometry.getSerialChildrenCount(); s++ ) {
            for ( int p = 0; p < geometry.getParallelChildrenCount(); p++ ) {
                Geometry child = geometry.getChild(s, p);
                child.setExpandedView(expandedView);
                internalConvert(child, expandedView);
                geometry.addChildGeometry(child, s, p);
            }
        }
    }
    
    
    private class InnerGeometry<K extends Geometry> {

        private final K geometry;
        private final Point point;

        InnerGeometry(K geometry, Point p) {
            this.geometry = geometry;
            this.point = p;
        }

        K getGeometry() {
            return geometry;
        }

        int getXPosition() {
            return point.x;
        }

        int getYPosition() {
            return point.y;
        }

        Point getPoint() {
            return point;
        }

    }

}
