package org.lsst.ccs.utilities.image;

import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.geometry.Geometry;
import org.lsst.ccs.utilities.ccd.CCD;

/**
 * An implementation of FitsHeaderMetadataProviderFactory based on a Geometry
 * object.
 * It internally creates CCD level FitsHeaderMetadataProvider objects that will
 * be used when fits files are written.
 * 
 * @author turri
 */
public class GeometryFitsHeaderMetadataProviderFactory implements FitsHeaderMetadataProviderFactory {

    private final Geometry geometry;
    private final Map<String,FitsHeaderMetadataProvider> providerMap = new HashMap<>();
    
    public GeometryFitsHeaderMetadataProviderFactory(Geometry geom) {
        this.geometry = geom;
        introspectGeometry(geometry);
        
    }
    
    @Override
    public FitsHeaderMetadataProvider createFitsHeaderMetadataProvider(ImageSet imageSet) {
        return providerMap.get(imageSet.getImageSetId());
    }
    
    
    private void introspectGeometry(Geometry<?> geom) {
        if ( geom instanceof CCD ) {
            createMetadataProvider((CCD) geom);
        } else {
            for ( Geometry g : geom.getChildrenList() ) {
                introspectGeometry(g);
            }
        }
    }
    
    private void createMetadataProvider(CCD ccd) {
        ImageSet imageSet = FitsHeaderUtilities.createImageSetForCCD(ccd);
        DefaultFitsHeaderMetadataProvider provider = new DefaultFitsHeaderMetadataProvider(imageSet);
        providerMap.put(ccd.getUniqueId(), provider);
    }
    
    
}
