package org.lsst.ccs.utilities.image;

import java.awt.Point;
import java.util.HashMap;
import java.util.Map;
import org.lsst.ccs.geometry.Geometry;
import org.lsst.ccs.utilities.ccd.CCDInterface;
import org.lsst.ccs.utilities.ccd.Raft;
import org.lsst.ccs.utilities.ccd.SegmentInterface;

/**
 * Utility class for the generation of header specifications objects.
 *
 * @author turri
 */
public class FitsHeaderUtilities {

    /**
     * Utility function to return a pair of values in Fits standard.
     *
     * @param a Lower edge
     * @param b Upper edge
     * @return Returns the par of values in Fits Standard.
     *
     */
    private static String pair(String a, String b) {
        return "[" + a + "," + b + "]";
    }

    /**
     * Utility function to return a range of values in Fits standard.
     *
     * @param a Lower edge
     * @param b Upper edge
     * @return Returns the range in Fits Standard.
     *
     */
    private static String range(int a, int b) {
        return a + ":" + b;
    }

    /**
     * Utility function to return a range of values in Fits standard.
     *
     * @param a Lower edge
     * @param b Upper edge
     * @param flip boolean to indicate if the range should be flipped.
     * @return Returns the range in Fits Standard.
     *
     */
    private static String range(int a, int b, boolean flip) {
        return flip ? range(b, a) : range(a, b);
    }

    /**
     * Create the default ImageSet for the given CCDInterface.
     * 
     * @param ccd The CCD for which to create the ImageSet
     * @return The corresponding ImageSet
     */
    public static ImageSet createImageSetForCCD(CCDInterface ccd) {
        Map<String, Object> metaData = new HashMap<>();
        metaData.putAll(getCCDPrimaryHeaders(ccd));
        DefaultImageSet imageSet = new DefaultImageSet(metaData);
        
        for (SegmentInterface segment : ccd.getSegments()) {
           MetaDataSet metaDataSet = new MetaDataSet();
            final Map<String, Object> imageMetaData = getSegmentHeaders(segment, false);
            metaDataSet.addMetaData("channel", imageMetaData);
            
            addRaftLevelMetadata(segment, metaDataSet, imageMetaData);

            imageSet.addImage(ccd.getType().getSegmentTotalSerialSize(), ccd.getType().getSegmentTotalParallelSize(), metaDataSet);
        }
        return imageSet;
    }
    
    private static void addRaftLevelMetadata(SegmentInterface segment, MetaDataSet imageMetaData, Map<String,Object> segmentHeaders) {
        Raft parentRaft = findRaftForGeometry((Geometry)segment);
        if ( parentRaft != null ) {
            Point raftPosition = parentRaft.getGeometryAbsolutePosition();            
            Point segmentPosition = ((Geometry)segment).getGeometryAbsolutePosition();
            
            int segmentXPositionInRaft = segmentPosition.x - raftPosition.x;
            int segmentYPositionInRaft = segmentPosition.y - raftPosition.y;
            
            Map<String, Object> extra = new HashMap<>();
            extra.put("WCSNAMEr", "RAFT");
            extra.put("CTYPE1r", "RAFT_X");
            extra.put("CTYPE2r", "RAFT_Y");
            extra.put("CUNIT1r", "pixel");
            extra.put("CUNIT2r", "pixel");
            extra.put("CD1_1r", segmentHeaders.get("CD1_1"));
            extra.put("CD1_2r", 0.0);
            extra.put("CD2_1r", 0.0);
            extra.put("CD2_2r", segmentHeaders.get("CD2_2"));
            extra.put("CRPIX1r", segmentXPositionInRaft);
            extra.put("CRPIX2r", segmentYPositionInRaft);
            imageMetaData.addMetaData("raft-wcs", extra);        
        }
    }
    
    private static Raft findRaftForGeometry(Geometry<?> geom) {
        Geometry parent = geom.getParent();
        if ( parent == null ) {
            return null;
        }
        if ( parent instanceof Raft ) {
            return (Raft)parent;
        }
        return findRaftForGeometry(parent);
    }
    
    /**
     * Create fits headers for the primary header for the given Object.
     *
     * @param ccd The CCD for which the primary header is returned.
     * @return A map containing the fits header names and values.
     */
    public static Map<String, Object> getCCDPrimaryHeaders(CCDInterface ccd) {
        Map<String, Object> primaryMetaData = new HashMap<>();
        primaryMetaData.put("DETSIZE", getCCDDetSize(ccd));
        return primaryMetaData;
    }

    /**
     * Create the DETSIZE fits header for the provided object.
     *
     * @param ccd The object for which the DETSIZE is to be evaluated.
     * @return The value for DETSIZE
     */
    private static String getCCDDetSize(CCDInterface ccd) {
        return pair(range(1, ccd.getActiveSerialSize()), FitsHeaderUtilities.range(1, ccd.getActiveParallelSize()));
    }

    /**
     * Create fits headers for this segment.
     *
     * @param seg
     * @return A map containing the fits header names and values.
     */
    public static Map<String, Object> getSegmentHeaders(SegmentInterface seg) {
        return getSegmentHeaders(seg, false);
    }

    /**
     * Create fits headers for a Segment. Optionally create distorted
     * headers so that over and underscan regions can be viewed in DS9, as
     * requested by users at BNL.
     *
     * @param segment The segment for which to build the Header.
     * @param treatOverAndUnderscanAsActive If <code>true</code> then the
     * DATASEC and DETSEC headers are modified to treat all pixels as
     * active, and the BIASSEC keyword is omitted. Note that files written
     * in this way are not compliant with LCA-10140.
     * @return A map containing the fits header names and values.
     */
    public static Map<String, Object> getSegmentHeaders(SegmentInterface segment, boolean treatOverAndUnderscanAsActive) {

        Map<String, Object> imageMetaData = new HashMap<>();

        imageMetaData.put("EXTNAME", String.format("Segment%01d%01d", getSegmentParallelPosition(segment), getSegmentSerialPosition(segment)));
        imageMetaData.put("CHANNEL", segment.getChannel());
        imageMetaData.put("DATASEC", getDataSec(segment, treatOverAndUnderscanAsActive));
        imageMetaData.put("DETSEC", getDetSec(segment, treatOverAndUnderscanAsActive));
        if (!treatOverAndUnderscanAsActive) {
            imageMetaData.put("BIASSEC", getBiasSec(segment));
        }
        imageMetaData.put("CCDSUM", "1 1");
        imageMetaData.put("LTV1", getLTV1(segment));
        imageMetaData.put("LTV2", getLTV2(segment));
        imageMetaData.put("LTM1_1", getLTM11(segment)); // delta along X axis
        imageMetaData.put("LTM2_2", getLTM22(segment)); // delta along Y axis
        //FIXME: Waiting for input from Jim and Seth
//            imageMetaData.put("DTV1", 0); // detector transformation vector
//            imageMetaData.put("DTV2", 0); // detector transformation vector
//            imageMetaData.put("DTM1_1", 1); // detector transformation matrix
//            imageMetaData.put("DTM2_2", 1); // detector transformation matrix

        // WCS Keywords (from Stuart)
        imageMetaData.put("WCSNAME", "PRIMARY");
        imageMetaData.put("CTYPE1", "CCD_X");
        imageMetaData.put("CTYPE2", "CCD_Y");
        imageMetaData.put("CUNIT1", "pixel");
        imageMetaData.put("CUNIT2", "pixel");
        imageMetaData.put("CD1_1", getCD11(segment));
        imageMetaData.put("CD1_2", 0.0);
        imageMetaData.put("CD2_1", 0.0);
        imageMetaData.put("CD2_2", getCD22(segment));
        imageMetaData.put("CRPIX1", getCRPIX1(segment));
        imageMetaData.put("CRPIX2", getCRPIX2(segment));
                
        return imageMetaData;
    }
    
    private static int getSegmentParallelPosition(SegmentInterface seg) {
        return seg.getParallelPosition();
    }
    private static int getSegmentSerialPosition(SegmentInterface seg) {
        return seg.getSerialPosition();
    }
    
    private static boolean isReadoutDown(SegmentInterface seg) {
        return seg.isReadoutDown();            
    }
    private static boolean isReadoutLeft(SegmentInterface seg) {
        return seg.isReadoutLeft();
    }

    /**
     * Generate the DETSEC fits header
     *
     * @return The value for DETSEC
     */
    private static String getDetSec(SegmentInterface segment, boolean treatOverAndUnderscanAsActive) {
        if (treatOverAndUnderscanAsActive) {
            return pair(range(segment.getSegmentTotalSerialSize() * getSegmentSerialPosition(segment) + 1,
                    segment.getSegmentTotalSerialSize() * (getSegmentSerialPosition(segment) + 1), !isReadoutLeft(segment)),
                    range(segment.getSegmentTotalParallelSize() * getSegmentParallelPosition(segment) + 1,
                            segment.getSegmentTotalParallelSize() * (getSegmentParallelPosition(segment) + 1),
                            !isReadoutDown(segment)));
        } else {
            return pair(range(segment.getSegmentSerialActiveSize() * getSegmentSerialPosition(segment) + 1,
                    segment.getSegmentSerialActiveSize() * (getSegmentSerialPosition(segment) + 1), !isReadoutLeft(segment)),
                    range(segment.getSegmentParallelActiveSize() * getSegmentParallelPosition(segment) + 1,
                            segment.getSegmentParallelActiveSize() * (getSegmentParallelPosition(segment) + 1),
                            !segment.isReadoutDown()));
        }
    }

    /**
     * Create the DATASEC fits header
     *
     * @return The value for DATASEC
     */
    private static String getDataSec(SegmentInterface segment, boolean treatOverAndUnderscanAsActive) {
        if (treatOverAndUnderscanAsActive) {
            return pair(range(1, segment.getSegmentTotalSerialSize()), range(1, segment.getSegmentTotalParallelSize()));
        } else {
            return pair(range(segment.getSegmentSerialPrescanSize() + 1, segment.getSegmentSerialActiveSize() + segment.getSegmentSerialPrescanSize()),
                    range(1, segment.getSegmentParallelActiveSize()));
        }
    }

    /**
     * Create the BIASSEC fits header
     *
     * @return The value for BIASSEC
     */
    private static String getBiasSec(SegmentInterface segment) {
        return pair(range(segment.getSegmentSerialActiveSize() + segment.getSegmentSerialPrescanSize() + 1,
                segment.getSegmentSerialActiveSize() + segment.getSegmentSerialPrescanSize() + segment.getSegmentSerialOverscanSize()),
                range(1, segment.getSegmentParallelActiveSize()));
    }

    private static double getLTV1(SegmentInterface seg) {
        return isReadoutDown(seg) ? seg.getSegmentSerialActiveSize() * (seg.getParallelPosition() + 1) : -(seg.getSegmentSerialActiveSize() * seg.getParallelPosition());
    }

    private static double getLTV2(SegmentInterface seg) {
        return (!isReadoutLeft(seg)) ? seg.getSegmentParallelActiveSize() * (seg.getSerialPosition() + 1) : seg.getSegmentParallelActiveSize() * seg.getSerialPosition();
    }

    private static double getLTM11(SegmentInterface seg) {
        return isReadoutDown(seg) ? -1.0 : 1.0;
    }

    private static double getLTM22(SegmentInterface seg) {
        return (!isReadoutLeft(seg)) ? -1.0 : 1.0;
    }

    private static double getCD11(SegmentInterface seg) {
        return !isReadoutLeft(seg) ? -1.0 : 1.0;
    }

    private static double getCD22(SegmentInterface seg) {
        return (!isReadoutDown(seg)) ? -1.0 : 1.0;
    }

    private static double getCRPIX1(SegmentInterface seg) {
        return (!isReadoutLeft(seg)) ? seg.getSegmentTotalSerialSize() * (getSegmentSerialPosition(seg) + 1) + 1 : -(seg.getSegmentTotalSerialSize() * getSegmentSerialPosition(seg));
    }

    private static double getCRPIX2(SegmentInterface seg) {
        return !isReadoutDown(seg) ? seg.getSegmentTotalParallelSize() * (getSegmentParallelPosition(seg) + 1) + 1 : seg.getSegmentTotalParallelSize() * getSegmentParallelPosition(seg);
    }

}
