package org.lsst.ccs.utilities.image;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Default implementation of the FitsHeaderMetadataProvider interface.
 * It defines internal structures and utility methods to fill them.
 *
 * @author turri
 */
public class DefaultFitsHeaderMetadataProvider implements FitsHeaderMetadataProvider {

    protected final Map<String, Map<String, Object>> primaryHeaderMetadata = new HashMap<>();
    protected final List<Map<String, Map<String, Object>>> extendedHeaderMetadataList;
    private final int numberOfExtendedDataHeaders;
    private final Object primaryLock = new Object();
    private final Object extendedLock = new Object();

    /**
     * Create a default FitsHeaderMetadataProvider for the given ImageSet.
     * At creation, if the ImageSet has metadata available, it will be added to
     * the primary header metadata.
     * Similarly each Image metadata, if available, will be added to the Data extended
     * headers metadata.
     *
     * @param imageSet The ImageSet for which this FitsHeaderMetadataProvider is created.
     *
     */
    public DefaultFitsHeaderMetadataProvider(ImageSet imageSet) {
        this.numberOfExtendedDataHeaders = imageSet.getImages().size();
        extendedHeaderMetadataList = new ArrayList<>(numberOfExtendedDataHeaders);

        //If available add the ImageSet metadata to the primary header
        Map<String, Object> imageSetMetadata = imageSet.getMetaData();
        if (imageSetMetadata != null) {
            primaryHeaderMetadata.put("imageSet", imageSetMetadata);
        }

        for (int i = 0; i < numberOfExtendedDataHeaders; i++) {
            extendedHeaderMetadataList.add(new HashMap<>());
            Map<String, Map<String, Object>> result = extendedHeaderMetadataList.get(i);
            Map<String, Map<String, Object>> imageMetadata = imageSet.getImages().get(i).getMetaData();
            if (imageMetadata != null) {
                result.putAll(imageMetadata);
            }
        }
    }

    @Override
    public Map<String, Map<String, Object>> getAdditionalExtendedHeaderMetadata(String extendedKeyword) {
        return getPrimaryHeaderMetadata();
    }

    @Override
    public Map<String, Map<String, Object>> getDataExtendedHeaderMetadata(int extendedIndex) {
        if (extendedIndex < 0 || extendedIndex >= numberOfExtendedDataHeaders) {
            throw new IllegalArgumentException("Wrong extended header data index " + extendedIndex + " must be positive and less than " + extendedIndex);
        }
        synchronized (extendedLock) {
            Map<String, Map<String, Object>> result = new HashMap<>();
            result.putAll(extendedHeaderMetadataList.get(extendedIndex));
            return result;
        }
    }

    @Override
    public Map<String, Map<String, Object>> getPrimaryHeaderMetadata() {
        synchronized (primaryLock) {
            Map<String, Map<String, Object>> result = new HashMap<>();
            result.putAll(primaryHeaderMetadata);
            return result;
        }
    }

    /**
     * Set a map in the primary Header metadata. This method replaces any existing metadata.
     *
     * @param primaryKey The Key in the primary header for the provided metadata.
     * @param map The map with the metadata to be added.
     */
    public void setPrimaryHeaderMetadata(String primaryKey, Map<String, Object> map) {
        synchronized (primaryLock) {
            primaryHeaderMetadata.put(primaryKey, map);
        }
    }

    /**
     * Set a map to a data extended header by index. This method replaces any existing metadata.
     *
     * @param extendedHeaderIndex The extended header index to which the provided metadata is set.
     * @param extendedHeaderKey The Key in the indexed extended data header to which the provided metadata is set.
     * @param map The map containing the metadata to be set.
     */
    public void setDataExtendedHeaderMetadata(int extendedHeaderIndex, String extendedHeaderKey, Map<String, Object> map) {
        synchronized (extendedLock) {
            extendedHeaderMetadataList.get(extendedHeaderIndex).put(extendedHeaderKey, map);
        }
    }

    /**
     * Add a map to the primary Header metadata. The provided map is added on top of any
     * existing map.
     *
     * @param primaryKey The Key in the primary header to which the provided metadata is added.
     * @param map The map with the metadata to be added.
     */
    public void addToPrimaryHeaderMetadata(String primaryKey, Map<String, Object> map) {
        synchronized (primaryLock) {
            Map<String, Object> existingMap = primaryHeaderMetadata.get(primaryKey);
            if (existingMap == null) {
                existingMap = new HashMap<>();
                primaryHeaderMetadata.put(primaryKey, existingMap);
            }
            existingMap.putAll(map);
        }
    }

    /**
     * Add a map to a data extended header by index. The provided map is added on top of any
     * existing map.
     *
     * @param extendedHeaderIndex The extended header index to add to.
     * @param extendedHeaderKey The Key in the indexed extended data header to which the provided metadata is added.
     * @param map The map containing the metadata to be added.
     */
    public void addToDataExtendedHeaderMetadata(int extendedHeaderIndex, String extendedHeaderKey, Map<String, Object> map) {
        synchronized (extendedLock) {
            Map<String, Object> existingMap = extendedHeaderMetadataList.get(extendedHeaderIndex).get(extendedHeaderKey);
            if (existingMap == null) {
                existingMap = new HashMap<>();
                extendedHeaderMetadataList.get(extendedHeaderIndex).put(extendedHeaderKey, existingMap);
            }
            existingMap.putAll(map);
        }
    }

}
