
package org.lsst.ccs.utilities.ccd.image.data;

import java.util.ArrayList;
import java.util.List;


/**
 * Interface for objects that provide DataStreams
 * 
 * @author turri
 */
public interface RawImageDataProvider {
    
    static List<InnerRawImageDataListener> listeners = new ArrayList<>();
    
    /**
     * Get the RawImageData for this object.
     * 
     * @return The RawImageData containing the data.
     */
    public RawImageData getRawImageData();
    
    
    /**
     * Add a RawImageDataListener for a given RawImagedataProvider.
     * 
     * @param listener The listener to add
     * @param provider
     */
    static void addRawImageDataListener(RawImageDataListener listener, RawImageDataProvider provider) {
        listeners.add(new InnerRawImageDataListener(listener,provider));
    }
    
    /**
     * Remove a RawImageDataListener to this RawImageDataProvider.
     * 
     * @param listener The listener to remove
     */
    static void removeRawImageDataListener(RawImageDataListener listener) {
        List<InnerRawImageDataListener> toRemove = new ArrayList<>();
        for ( InnerRawImageDataListener inner : listeners ) {
            if ( inner.getListener() == listener ) {
                toRemove.add(inner);
            }
        }
        for ( InnerRawImageDataListener inner : toRemove ) {
            listeners.remove(inner);            
        }
    }
    
    /**
     * Notify the listeners that RawImageData is available for a given RawImageDataProvider.
     * 
     * @param dataProvider the RawImageDataProvider for which the data is available.
     */
    static void notifyRawImageDataListeners(RawImageDataProvider dataProvider) {
        for ( InnerRawImageDataListener l : listeners ) {
            if ( l.getProvider() == dataProvider ) {
                l.getListener().rawImageDataAvailable(dataProvider);
            }
        }
    }
    
    static class InnerRawImageDataListener {
        final RawImageDataListener listener;
        final RawImageDataProvider provider;
        
        InnerRawImageDataListener(RawImageDataListener listener, RawImageDataProvider provider) {
            this.listener = listener;
            this.provider = provider;
        }
        
        RawImageDataListener getListener() {
            return listener;
        }
        
        RawImageDataProvider getProvider() {
            return provider;
        }
    }
}
