package org.lsst.ccs.utilities.ccd;

import java.awt.Point;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import org.lsst.ccs.geometry.Geometry;
import org.lsst.ccs.geometry.ImageSensitiveArea;
import org.lsst.ccs.utilities.ccd.image.data.RawImageDataProvider;
import org.lsst.ccs.utilities.ccd.image.data.RawImageData;
import org.lsst.ccs.utilities.image.ImagePixelData;

/**
 * An class to represent a CCD Segment.
 *
 * @author turri
 */
public class Segment extends Geometry implements RawImageDataProvider, ImageSensitiveArea, SegmentInterface {

    private final int serialPrescan, serialOverscan, parallelOverscan;
    private final SegmentReadOutPoint readoutPoint;
    private ImagePixelData imagePixelData;
    private final int serialActive, parallelActive;
    private final int serialPosition, parallelPosition;
    private final int channel;

    /**
     * CCDSegment constructor.
     *
     * @param serialActive Length of the portion of the serial register that receives signal from the array.
     * @param parallelActive Number of light sensitive rows in the CCDSegment.
     * @param serialPrescan Number of prescan pixels in the serial register.
     * @param serialOverscan Number of additional phantom pixels that are read out from the serial register.
     * @param parallelOverscan Number of additional phantom rows that are read from the CCDSegment.
     */
    Segment(CCDType ccdType, String name, int serialPosition, int parallelPosition, int channel) {
        super(name);
        this.serialPrescan = ccdType.getSegmentSerialPrescanSize();
        this.serialOverscan = ccdType.getSegmentSerialOverscanSize();
        this.parallelOverscan = ccdType.getSegmentParallelOverscanSize();
        this.serialActive = ccdType.getSegmentSerialActiveSize();
        this.parallelActive = ccdType.getSegmentParallelActiveSize();
        this.serialPosition = serialPosition;
        this.parallelPosition = parallelPosition;
        setExpandedView(false);
        if (parallelPosition == 1) {
            readoutPoint = SegmentReadOutPoint.RightUp;
        } else {
            readoutPoint = ccdType.name().toLowerCase().contains("e2v") ? SegmentReadOutPoint.LeftDown : SegmentReadOutPoint.RightDown;
        }
        this.channel = channel;
    }

    void setRedout(boolean isTopRow, String CCDtype) {
        if (readoutPoint != null) {
            throw new RuntimeException("The Segment Readout point can only be set once!");
        }
    }

    protected void setExpandedView(boolean expandedView) {
        if (expandedView) {
            setDimension(getSegmentTotalSerialSize(), getSegmentTotalParallelSize());
        } else {
            setDimension(serialActive, parallelActive);
        }

    }

    @Override
    public int getChannel() {
        return channel;
    }

    @Override
    public int getParallelPosition() {
        return parallelPosition;
    }

    @Override
    public int getSerialPosition() {
        return serialPosition;
    }

    public SegmentReadOutPoint getSegmentReadOutPoint() {
        return readoutPoint;
    }

    @Override
    public int getSegmentSerialPrescanSize() {
        return serialPrescan;
    }

    @Override
    public int getSegmentSerialOverscanSize() {
        return serialOverscan;
    }

    @Override
    public int getSegmentParallelOverscanSize() {
        return parallelOverscan;
    }

    @Override
    public int getSegmentSerialActiveSize() {
        return serialActive;
    }

    @Override
    public int getSegmentParallelActiveSize() {
        return parallelActive;
    }

    @Override
    public int getSegmentTotalSerialSize() {
        return getSegmentSerialPrescanSize() + getSegmentSerialActiveSize() + getSegmentSerialOverscanSize();
    }

    @Override
    public int getSegmentTotalParallelSize() {
        return getSegmentParallelActiveSize() + getSegmentParallelOverscanSize();
    }

    @Override
    public boolean isReadoutDown() {
        return readoutPoint.isParallelDown();
    }

    @Override
    public boolean isReadoutLeft() {
        return readoutPoint.isSerialLeft();
    }

    @Override
    public CCDInterface getParentCCD() {
        return (CCDInterface)getParent();
    }

    
    
    /**
     * Get the buffer of image data accumulated by this Segment.
     *
     * @return The ByteBuffer with the data;
     */
    @Override
    public RawImageData getRawImageData() {
        DataStreamer streamer = new DataStreamer(this);
        ByteBuffer dest = createByteBuffer(getSegmentTotalSerialSize() * getSegmentTotalParallelSize());
        for (int row = 0; row < getSegmentParallelActiveSize(); row++) {
            streamer.fillSerialPrescan(dest);
            streamer.fillDataRow(dest);
            streamer.fillSerialOverscan(dest);
        }
        streamer.fillParallelOverscan(dest);
        dest.flip();
        return new RawImageData(RawImageData.BitsPerPixel.BIT32, dest);
    }

    private class DataStreamer {

        private final ByteBuffer serialPrescanBuffer;
        private final ByteBuffer serialOverscanBuffer;
        private final ByteBuffer parallelOverscanBuffer;
        private final ImagePixelData imagePixelData;
        private int currentDataRow;
        private boolean increaseAlongParallel;
        private boolean scanRowFromTop;
        private final Segment s;

        DataStreamer(Segment s) {
            serialPrescanBuffer = createByteBuffer(s.getSegmentSerialPrescanSize());
            serialOverscanBuffer = createByteBuffer(s.getSegmentSerialOverscanSize());
            parallelOverscanBuffer = createByteBuffer(s.getSegmentParallelOverscanSize() * s.getSegmentTotalSerialSize());
            for (int i = 0; i < s.getSegmentSerialPrescanSize(); i++) {
                serialPrescanBuffer.putInt(0);
            }
            for (int i = 0; i < s.getSegmentSerialOverscanSize(); i++) {
                serialOverscanBuffer.putInt(0);
            }
            for (int i = 0; i < s.getSegmentParallelOverscanSize() * s.getSegmentTotalSerialSize(); i++) {
                parallelOverscanBuffer.putInt(0);
            }
            imagePixelData = s.getImagePixelData();

            this.s = s;
            reset();
        }

        public final void reset() {
            SegmentReadOutPoint readout = s.readoutPoint;
            if (readout.isParallelDown()) {
                currentDataRow = imagePixelData.getHeight() - 1;
                increaseAlongParallel = false;
            } else {
                currentDataRow = 0;
                increaseAlongParallel = true;
            }
            scanRowFromTop = !readout.isSerialLeft();
        }

        void fillSerialPrescan(ByteBuffer data) {
            serialPrescanBuffer.flip();
            data.put(serialPrescanBuffer);
        }

        void fillSerialOverscan(ByteBuffer data) {
            serialOverscanBuffer.flip();
            data.put(serialOverscanBuffer);
        }

        void fillParallelOverscan(ByteBuffer data) {
            parallelOverscanBuffer.flip();
            data.put(parallelOverscanBuffer);
        }

        void fillDataRow(ByteBuffer data) {
            if (scanRowFromTop) {
                for (int i = imagePixelData.getWidth() - 1; i >= 0; i--) {
                    data.putInt(imagePixelData.getPixelData(i, currentDataRow));
                }
            } else {
                for (int i = 0; i < imagePixelData.getWidth(); i++) {
                    data.putInt(imagePixelData.getPixelData(i, currentDataRow));
                }
            }
            currentDataRow = increaseAlongParallel ? currentDataRow + 1 : currentDataRow - 1;
        }
    }

    private ByteBuffer createByteBuffer(int size) {
        ByteBuffer dest = ByteBuffer.allocateDirect(size * 4);
        dest.order(ByteOrder.nativeOrder());
        return dest;
    }

    /**
     * The Segment readout point with respect to the serial and parallel direction.
     * The canonical readout point when dealing with fits files is LeftDown; any
     * other readout point will need some sort of transformation.
     *
     */
    public enum SegmentReadOutPoint {

        LeftDown(true, true), LeftUp(true, false), RightDown(false, true), RightUp(false, false);

        private final boolean serialLeft, parallelDown;

        private SegmentReadOutPoint(boolean serialLeft, boolean parallelDown) {
            this.serialLeft = serialLeft;
            this.parallelDown = parallelDown;
        }

        public boolean isParallelDown() {
            return parallelDown;
        }

        public boolean isSerialLeft() {
            return serialLeft;
        }

    };

    /**
     * Utility function to create CCDSegments by type.
     *
     * @param ccdType The type of the CCD.
     * @param label The label of the segment.
     *
     * @return The Segment for the provided type.
     */
    public static Segment createCCDSegment(CCDType ccdType, String label, int serialPosition, int parallelPosition, int channel) {
        return new Segment(ccdType, label, serialPosition, parallelPosition, channel);
    }

    @Override
    public void exposeToImage(ImagePixelData imagePixelData) {
        Point absoluteOrigin = getAbsolutePoint(new Point(0, 0));
        this.imagePixelData = new SegmentPixelData(absoluteOrigin, getWidth(), getHeight(), imagePixelData);
    }

    @Override
    public boolean hasPixelData() {
        return imagePixelData != null;
    }

    @Override
    public ImagePixelData getImagePixelData() {
        return imagePixelData;
    }

    /**
     * This class extracts the Segment specific data from the overall image
     * to which this Geometry is exposed.
     * It uses the global position of this Geometry in the outermost geometry,
     * its width and height to extract the sub-region of interest.
     *
     */
    private class SegmentPixelData implements ImagePixelData {

        private final int xGlobalCoordinate, yGlobalCoordinate;
        private final int width, height;
        private final ImagePixelData exposure;

        SegmentPixelData(Point origin, int width, int height, ImagePixelData exposure) {
            this.xGlobalCoordinate = origin.x;
            this.yGlobalCoordinate = origin.y;
            this.width = width;
            this.height = height;
            this.exposure = exposure;
        }

        @Override
        public int getHeight() {
            return height;
        }

        @Override
        public int getWidth() {
            return width;
        }

        @Override
        public int getPixelData(int x, int y) {
            return exposure.getPixelData(x += xGlobalCoordinate, y += yGlobalCoordinate);
        }

        @Override
        public double getMaxValue() {
            return exposure.getMaxValue();
        }
    }

}
