package org.lsst.ccs.utilities.ccd;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.util.ArrayList;
import java.util.List;
import org.lsst.ccs.geometry.Geometry;
import org.lsst.ccs.utilities.ccd.image.data.RawImageData;
import org.lsst.ccs.utilities.ccd.image.data.RawImageDataProvider;

public class Reb extends Geometry<CCD> implements RawImageDataProvider {

    private final int sensorGap = 10;
    private final static RawImageData.BitsPerPixel bits = RawImageData.BitsPerPixel.BIT32;

    public Reb(String name) {
        super(name, 3, 1);
    }

    @Override
    protected void addGeometryToGrid(CCD child, int s, int p) {
        if (p != 0) {
            throw new IllegalArgumentException("Rebs only allow one row of CCDs in it");
        }
        int xCoord = s * child.getWidth();
        xCoord += s * sensorGap;
        addGeometry(child, xCoord, 0);
    }

    public static Reb createReb(String label, int row, CCDType type) {
        Reb reb = new Reb(label);
        //Add three CCDs left to right
        for (int i = 0; i < 3; i++) {
            String ccdLabel = "Sen" + row + i;
            CCD ccd = CCD.createCCD(ccdLabel, type);
            reb.addChildGeometry(ccd, i, 0);
        }
        return reb;
    }

    //TO DO: Do we need to copy the data there?
    //Please notice that the format for the output data is
    //in buckets of corresponding segment pixels, i.e.
    //the format the DAQ is providing. This should NOT be
    //done here, but somewhere else where the DAQ is simulated.
    @Override
    public RawImageData getRawImageData() {
        int totalSize = 0;
        List<ByteBuffer> availableData = new ArrayList<>();
        int segCount = 0;
        for (int s = 0; s < getSerialChildrenCount(); s++) {
            CCD ccd = getChild(s, 0);
            for (SegmentInterface segment : ccd.getSegments()) {
                RawImageData imageData = ((Segment) segment).getRawImageData();
                availableData.add(imageData.getImageData());
                totalSize += imageData.getImageData().limit();
                segCount++;
            }
        }

        
        
        ByteBuffer result = ByteBuffer.allocateDirect(totalSize);
        result.order(ByteOrder.nativeOrder());
        int segmentData = totalSize/segCount/bits.bytes();
        for (int i = 0; i < segmentData; i++) {
            for (ByteBuffer byteBuffer : availableData) {
                result.putInt(byteBuffer.getInt());
            }
        }

        result.flip();
        return new RawImageData(bits, result);
    }

}
