package org.lsst.ccs.utilities.ccd;

import org.lsst.ccs.utilities.ccd.CCDGeometry.ReadoutOrder;

/**
 * An enumeration of standard camera CCD types. Can be used to get the geometry of a CCD.
 * @author tonyj
 */
public enum CCDType {

    E2V(10, 2002, 512, 22, 46, true), ITL(3, 2000, 509, 32, 48, false),
    TEST_SMALL_E2V(1, 200, 50, 10, 4, true), TEST_SMALL_ITL(1, 200, 50, 10, 5, false);
    
    private CCDGeometry ccdGeom;
    private static final int ROW_COUNT = 8;
    private static final int COLUMN_COUNT = 2;
    private final int serialActiveSize, parallelActiveSize, serialPrescanSize, serialOverscanSize, parallelOverscanSize;

    private CCDType(int prescan, int segmentWidth, int segmentHeight,
            int overscan, int parallelOverscan, boolean isE2V) {        
        this.serialActiveSize = segmentHeight;
        this.parallelActiveSize = segmentWidth;
        this.serialPrescanSize = prescan;
        this.serialOverscanSize = overscan;
        this.parallelOverscanSize = parallelOverscan;

        ccdGeom = new CCDGeometry(ROW_COUNT, COLUMN_COUNT, segmentHeight, segmentWidth, prescan,
                overscan, parallelOverscan, this);
        
        for (int i = 0; i < 8; i++) {
            ccdGeom.addSegment(i + 1, i, 1, ReadoutOrder.UpRight);
        }
        for (int i = 8; i < 16; i++) {
            ccdGeom.addSegment(i + 1, 15-i, 0, isE2V ? ReadoutOrder.DownLeft : ReadoutOrder.DownRight);
        }
    }
    
    /**
     * Get the Segment total serial size: prescan, overscan and active size.
     * @return The Segment total serial size.
     */
    public int getSegmentTotalSerialSize() {
        return getSegmentSerialActiveSize()+getSegmentSerialOverscanSize()+getSegmentSerialPrescanSize();
    }

    /**
     * Get the Segment serial active size.
     * @return The Segment serial active size.
     */
    public int getSegmentSerialActiveSize() {
        return serialActiveSize;
    }
    
    /**
     * Get the Segment serial prescan size.
     * @return The Segment serial prescan size.
     */
    public int getSegmentSerialPrescanSize() {
        return serialPrescanSize;
    }

    /**
     * Get the Segment serial overscan size.
     * @return The Segment serial overscan size.
     */
    public int getSegmentSerialOverscanSize() {
        return serialOverscanSize;
    }

    /**
     * Get the Segment total parallel size: overscan and active size.
     * @return The Segment total parallel size.
     */
    public int getSegmentTotalParallelSize() {
        return getSegmentParallelActiveSize()+getSegmentParallelOverscanSize();
    }

    /**
     * Get the Segment parallel active size.
     * @return The Segment parallel active size.
     */
    public int getSegmentParallelActiveSize() {
        return parallelActiveSize;
    }
    
    /**
     * Get the Segment parallel overscan size.
     * @return The Segment parallel overscan size.
     */
    public int getSegmentParallelOverscanSize() {
        return parallelOverscanSize;
    }

    /**
     * Get the geometry for this CCD.
     * @return The geometry corresponding to this CCD type.
     */
    public CCDGeometry getGeometry() {
        return ccdGeom;
    }

};
