package org.lsst.ccs.utilities.ccd;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Defines the geometry of a CCD. See LCA-10103, in particular the diagram on
 * page 7.
 *
 * @see <a href="http://ls.st/lca-10103" target="_top">LCA-10103</a>
 */
public class CCDGeometry implements CCDInterface {

    private final int segmentRowCount;
    private final int segmentColumnCount;
    private final int serialPrescan;
    private final int parallelActive;
    private final int serialActive;
    private final int serialOverscan;
    private final int parallelOverscan;
    private final List<CCDSegment> segments;
    private final CCDSegment[] channelMap;
    private final CCDType type;

    /**
     * Create a CCDGeomtry object. To create geometry objects for standard CCD
     * types see the {@link CCDType} class.
     *
     * @param segmentRowCount The number of segments in the horizonal (serial)
     * direction
     * @param segmentColumnCount The number of segments in the vertical
     * (parallel) direction
     * @param serialActive The number of active pixels in the serial direction
     * @param parallelActive The number active pixels in the parallel direction
     * @param serialPrescan The number of serial prescan pixels for each
     * segment.
     * @param serialOverscan The number of overscan pixels in the serial
     * direction
     * @param parallelOverscan The number of overscan pixels in the parallel
     * direction
     */
    public CCDGeometry(int segmentRowCount, int segmentColumnCount, int serialActive, int parallelActive, int serialPrescan, int serialOverscan, int parallelOverscan, CCDType type) {
        this.segmentRowCount = segmentRowCount;
        this.segmentColumnCount = segmentColumnCount;
        this.serialPrescan = serialPrescan;
        this.parallelActive = parallelActive;
        this.serialActive = serialActive;
        this.serialOverscan = serialOverscan;
        this.parallelOverscan = parallelOverscan;
        this.segments = new ArrayList<>(segmentRowCount * segmentColumnCount);
        this.channelMap = new CCDSegment[segmentRowCount * segmentColumnCount];
        this.type = type;
    }

    public CCDGeometry getROIGeometry(int rows, int cols, int overRows, int overCols) {
        CCDGeometry g = new CCDGeometry(segmentRowCount, segmentColumnCount, cols, rows, 0, overCols, overRows, type);

        for (CCDSegment seg : segments) {
            g.addSegment(seg.channel, seg.row, seg.column, seg.readout);
        }

        return g;
    }

    public int getSerialPrescanCount() {
        return serialPrescan;
    }

    public int getParallelActiveCount() {
        return parallelActive;
    }

    public int getSerialActiveCount() {
        return serialActive;
    }

    public int getSerialOverscanCount() {
        return serialOverscan;
    }

    public int getParallelOverscanCount() {
        return parallelOverscan;
    }

    /**
     * Total serial pixels per segment.
     *
     * @return The total number of active+prescan+overscan pixels in the serial
     * direction.
     */
    public int getTotalSerialCount() {
        return getSerialPrescanCount() + getSerialActiveCount() + getSerialOverscanCount();
    }

    /**
     * Total parallel pixels per segment.
     *
     * @return The total number of active+overscan pixels in the parallel
     * direction.
     */
    public int getTotalParallelCount() {
        return getParallelActiveCount() + getParallelOverscanCount();
    }

    @Override
    public int getTotalSerialSize() {
        return getTotalSerialCount() * getSerialSegmentCount();
    }

    @Override
    public int getActiveSerialSize() {
        return getSerialActiveCount() * getSerialSegmentCount();
    }

    @Override
    public int getTotalParallelSize() {
        return getTotalParallelCount() * getParallelSegmentCount();
    }

    @Override
    public int getActiveParallelSize() {
        return getParallelActiveCount() * getParallelSegmentCount();
    }

    @Override
    public CCDType getType() {
        return type;
    }

    /**
     * Get the CCDSegment corresponding to the given row and column
     *
     * @param row The input row
     * @param column The input column
     * @return The CCDSegment and the specified row and column.
     */
    @Override
    public SegmentInterface getSegment(int row, int column) {
        assert (row >= 0 && row < segmentRowCount);
        assert (column >= 0 && column < segmentColumnCount);
        return channelMap[column * segmentRowCount + row];
    }

    @Override
    public List<SegmentInterface> getSegments() {
        return Collections.unmodifiableList(segments);
    }

    /**
     * Used when constructing the geometry to add segments (amplifiers) to the
     * geometry.
     *
     * @param channel The channel number (amplifier number) of the segment
     * (numbered from 1)
     * @param row The row containing the segment, numbered from 0 in the +Y
     * direction
     * @param column The column containing the segment, numbered from 0 in the
     * +X direction
     * @param readoutOrder The readout direction for the amplifier.
     */
    public void addSegment(int channel, int row, int column, ReadoutOrder readoutOrder) {
        final CCDSegment segment = new CCDSegment(channel, row, column, readoutOrder);
        segments.add(segment);
        channelMap[column * segmentRowCount + row] = segment;
    }

    @Override
    public int getParallelSegmentCount() {
        return segmentColumnCount;
    }
    
    @Override
    public int getSerialSegmentCount() {
        return segmentRowCount;
    }




    /**
     * Define the readout order for a particular amplifier. In this context
     * <ul>
     * <li>Left means +x</li>
     * <li>Right means -x</li>
     * <li>Up means -y</li>
     * <li>Down means +y</li>
     * </ul>
     * So DownRight means that the serial is readout Down (+y) and the parallel
     * is readout Right (-x).
     */
    public enum ReadoutOrder {

        DownRight(true, true), UpRight(false, true), DownLeft(true, false), UpLeft(false, false);

        private boolean down;
        private boolean right;

        private ReadoutOrder(boolean down, boolean right) {
            this.down = down;
            this.right = right;
        }

        public boolean isDown() {
            return down;
        }

        public boolean isRight() {
            return right;
        }

    };

    /**
     * Class representing a single segment (amplifier) within a CCD.
     */
    public class CCDSegment implements SegmentInterface {

        private ReadoutOrder readout;
        private int channel;
        private int row;
        private int column;

        private CCDSegment(int channel, int row, int column, ReadoutOrder readout) {
            this.readout = readout;
            this.channel = channel;
            this.row = row;
            this.column = column;
        }

        @Override
        public int getParallelPosition() {
            return column;
        }

        @Override
        public int getSerialPosition() {
            return row;
        }

        @Override
        public int getChannel() {
            return channel;
        }

        @Override
        public int getSegmentTotalSerialSize() {
            return getTotalSerialCount();
        }

        @Override
        public int getSegmentSerialActiveSize() {
            return getSerialActiveCount();
        }

        @Override
        public int getSegmentSerialPrescanSize() {
            return getSerialPrescanCount();
        }

        @Override
        public int getSegmentSerialOverscanSize() {
            return getSerialOverscanCount();
        }

        @Override
        public int getSegmentTotalParallelSize() {
            return getTotalParallelCount();
        }

        @Override
        public int getSegmentParallelActiveSize() {
            return getParallelActiveCount();
        }

        @Override
        public int getSegmentParallelOverscanSize() {
            return getParallelOverscanCount();
        }

        @Override    
        public boolean isReadoutDown() {
            return readout.isDown();
        }

        @Override
        public boolean isReadoutLeft() {
            return ! readout.isRight();
        }
        
        

        public ReadoutOrder getReadout() {
            return readout;
        }

        public int getRow() {
            return row;
        }

        public int getColumn() {
            return column;
        }

        /**
         * Get the CCD geometry of which the segment is a part
         *
         * @return The CCDGeometry
         */
        public CCDGeometry getParentCCD() {
            return CCDGeometry.this;
        }

        @Override
        public String toString() {
            return "CCDSegment{" + "readout=" + readout + ", channel=" + channel + ", row=" + row + ", column=" + column + '}';
        }

    }
}
