
package org.lsst.ccs.geometry;

import java.awt.Color;
import java.awt.Graphics;
import java.awt.Point;
import java.util.Map;
import org.lsst.ccs.utilities.ccd.CCD;
import org.lsst.ccs.utilities.ccd.Raft;
import org.lsst.ccs.utilities.ccd.Reb;
import org.lsst.ccs.utilities.ccd.Segment;

/**
 * A class dedicated to rendering Geometries to screen.
 * This class contains the awt dependencies so that Geometry classes don't have to.
 * 
 * @author turri
 */
public class GeometryRenderer {
    
    /**
     * Paint a Geometry with respect to the Geometry absolute position
     * within the outermost geometry. The provided point is the upper left corner
     * of this Geometry in the reference system of the outermost Geometry.
     *
     * @param geometry The Geometry to be rendered
     * @param g The graphics element
     * @param p The absolute point.
     *
     */
    public static void renderGeometry(Geometry<?> geometry, Graphics g, Point p) {
        renderGeometrySpecificDetails(geometry, g,p);
        if (geometry.hasChildren()) {
            Map<Geometry, Point> children = geometry.getChildrenWithAbsoluteCoordinates();
            for (Geometry child : children.keySet()) {
                renderGeometry(child, g, children.get(child));
            }
        }
    }
    
    private static void renderGeometrySpecificDetails(Geometry<?> geom, Graphics g, Point p) {
        if ( geom instanceof CCD ) {
            renderCCDGeometry((CCD)geom, g, p);
        } else if ( geom instanceof Raft ) {
            renderRaftGeometry((Raft)geom, g, p);
        } else if ( geom instanceof Reb ) {
            renderRebGeometry((Reb)geom, g, p);
        } else if ( geom instanceof Segment ) {
            renderSegmentGeometry((Segment)geom, g, p);
        }
        
    }
    
    private static void renderCCDGeometry(CCD geom, Graphics g, Point p) {
        g.drawRect(p.x, p.y, geom.getWidth(), geom.getHeight());

        Color orig = g.getColor();
        Color blue = Color.BLUE;
        g.setColor(blue);
        if ( ! geom.getName().isEmpty() ) {
            int textHeight = g.getFontMetrics().getHeight();
            int textWidth = g.getFontMetrics().stringWidth(geom.getName());
                
            int xLabel = p.x+(geom.getWidth()-textWidth)/2;       
            int yLabel = p.y+geom.getHeight()/2 + textHeight +2 ;        
        
            g.drawString(geom.getName(), xLabel, yLabel);        
        }        
        
        g.setColor(orig);
    }
    
    private static void renderRaftGeometry(Raft geom, Graphics g, Point p) {
        
        Color orig = g.getColor();
        Color green = Color.ORANGE;
        g.setColor(green);
        
        g.drawRect(p.x, p.y, geom.getWidth(), geom.getHeight());
        
        if ( ! geom.getName().isEmpty() ) {
            int xLabel = p.x + (int)(geom.getWidth()*.05);       
            int yLabel = p.y + (int)(geom.getHeight()*.05);        
        
            g.drawString(geom.getName(), xLabel, yLabel);        
        }        
        
        g.setColor(orig);
    }
    
    private static void renderRebGeometry(Reb geom, Graphics g, Point p) {
        
        int rebLineY = p.y + (int)(geom.getHeight()*.2);
        int lineWidth = (int)(geom.getWidth()*.85);
        int rebLineX = p.x + (int)((geom.getWidth()-lineWidth)/2);

        Color orig = g.getColor();
        Color green = Color.GREEN;
        g.setColor(green);
        
        g.fillRect(rebLineX, rebLineY, lineWidth, 4);
        
        if ( ! geom.getName().isEmpty() ) {
            int textHeight = g.getFontMetrics().getHeight();
                
            int xLabel = rebLineX+3;       
            int yLabel = rebLineY+3+textHeight;        
        
            g.drawString(geom.getName(), xLabel, yLabel);        
        }        
        
        g.setColor(orig);
    }
    
    private static void renderSegmentGeometry(Segment geom, Graphics g, Point p) {
        int width = geom.getWidth();
        int height = geom.getHeight();
        int xLabel = (int)(width*.5);
        int yLabel = (int)(width*.25);        
        int textHeight = g.getFontMetrics().getHeight();
        int textWidth = g.getFontMetrics().stringWidth(geom.getName());

        g.drawRect(p.x, p.y, geom.getWidth(), geom.getHeight());
        
        Color old = g.getColor();
        Color red = Color.RED;
        g.setColor(red);
        
        int squareSize = 4;
        
        if ( geom.getSegmentReadOutPoint().isParallelDown() ) {
            yLabel = p.y + height - yLabel;            
            if ( !geom.getSegmentReadOutPoint().isSerialLeft() ) {
                //Bottom right corner
                g.fillRect(p.x+geom.getWidth()-squareSize, p.y + height -squareSize, squareSize, squareSize);
            } else {
                //Bottom left corner
                g.fillRect(p.x, p.y + height-squareSize, squareSize, squareSize);
            }
        } else {
            yLabel = p.y + yLabel + textHeight;            
            if ( !geom.getSegmentReadOutPoint().isSerialLeft() ) {
                //Top right corner
                g.fillRect(p.x+geom.getWidth()-squareSize, p.y, squareSize, squareSize);
            } else {
                //Top left corner
                g.fillRect(p.x, p.y , squareSize, squareSize);
            }            
        }
        g.setColor(old);

        xLabel = p.x + xLabel - textWidth/2;
        g.drawString(geom.getName(), xLabel, yLabel);
               
        
    }
    
}
