package org.lsst.ccs.utilities.ccd;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Defines the geometry of a CCD. See LCA-10103, in particular the diagram on
 * page 7.
 *
 * @see <a href="http://ls.st/lca-10103" target="_top">LCA-10103</a>
 */
public class CCDGeometry {

    private final int segmentRowCount;
    private final int segmentColumnCount;
    private final int serialPrescan;
    private final int parallelActive;
    private final int serialActive;
    private final int serialOverscan;
    private final int parallelOverscan;
    private final List<CCDSegment> segments;
    private final CCDSegment[] channelMap;
    private final boolean isE2V;

    /**
     * Create a CCDGeomtry object. To create geometry objects for standard CCD
     * types see the {@link CCDType} class.
     *
     * @param segmentRowCount The number of segments in the horizonal (serial)
     * direction
     * @param segmentColumnCount The number of segments in the vertical
     * (parallel) direction
     * @param serialActive The number of active pixels in the serial direction
     * @param parallelActive The number active pixels in the parallel direction
     * @param serialPrescan The number of serial prescan pixels for each
     * segment.
     * @param serialOverscan The number of overscan pixels in the serial
     * direction
     * @param parallelOverscan The number of overscan pixels in the parallel
     * direction
     * @param isE2V Determines if the second row readout is reversed or not
     */
    public CCDGeometry(int segmentRowCount, int segmentColumnCount, int serialActive, int parallelActive, int serialPrescan, int serialOverscan, int parallelOverscan, boolean isE2V) {
        this.segmentRowCount = segmentRowCount;
        this.segmentColumnCount = segmentColumnCount;
        this.serialPrescan = serialPrescan;
        this.parallelActive = parallelActive;
        this.serialActive = serialActive;
        this.serialOverscan = serialOverscan;
        this.parallelOverscan = parallelOverscan;
        this.segments = new ArrayList<>(segmentRowCount * segmentColumnCount);
        this.channelMap = new CCDSegment[segmentRowCount * segmentColumnCount];
        this.isE2V = isE2V;
        
        if (segmentColumnCount != 2) {
            throw new IllegalArgumentException("Only segmentColumnCount==2 is currently supported");
        }

        for (int i = 0; i < segmentRowCount; i++) {
            addSegment(i + 1, i, 1, ReadoutOrder.DownRight);
        }
        for (int i = segmentRowCount; i < 2*segmentRowCount; i++) {
            addSegment(i + 1, 15-i, 0, isE2V ? ReadoutOrder.UpLeft : ReadoutOrder.DownLeft);
        }
    }
    
    /**
     * Returns a modified geometry with the serial overscan changed to the given value.
     * @param serialOverscan The number of overscan pixels in the serial direction.
     * @return The modified geometry.
     */
    public CCDGeometry withOverscan(int serialOverscan) {
        return new CCDGeometry(segmentRowCount,segmentColumnCount,serialActive,parallelActive,serialPrescan,serialOverscan,parallelOverscan,isE2V);
    }
    
    /**
     * Returns a modified geometry with the specified ROI. Note this method has no test
     * cases and it is hard to be sure it really functions the way it was intended. Caller beware.
     * @param rows
     * @param cols
     * @param overRows
     * @param overCols
     * @return 
     */
    public CCDGeometry withROIGeometry(int rows, int cols, int overRows, int overCols) {
        CCDGeometry g =  new CCDGeometry(segmentRowCount, segmentColumnCount, cols, rows, 0, overCols, overRows, isE2V);
        // FIXME: This code has not been tested/verified since fix for LSSTCCS-949.
        g.segments.clear();
        for (CCDSegment seg : segments) {
            g.addSegment(seg.channel, seg.row, seg.column, seg.readout);
        }
        
        return g;
    }

    public int getSerialPrescanCount() {
        return serialPrescan;
    }

    public int getParallelActiveCount() {
        return parallelActive;
    }

    public int getSerialActiveCount() {
        return serialActive;
    }

    public int getSerialOverscanCount() {
        return serialOverscan;
    }

    public int getParallelOverscanCount() {
        return parallelOverscan;
    }

    public int getSegmentRowCount() {
        return segmentRowCount;
    }

    public int getSegmentColumnCount() {
        return segmentColumnCount;
    }

    /**
     * Total serial pixels per segment.
     *
     * @return The total number of active+prescan+overscan pixels in the serial
     * direction.
     */
    public int getTotalSerialCount() {
        return getSerialPrescanCount() + getSerialActiveCount() + getSerialOverscanCount();
    }

    /**
     * Total parallel pixels per segment.
     *
     * @return The total number of active+overscan pixels in the parallel
     * direction.
     */
    public int getTotalParallelCount() {
        return getParallelActiveCount() + getParallelOverscanCount();
    }

    /**
     * The total height of the CCD in pixels
     *
     * @return The total height in pixels, including active+prescan+overscan
     * pixels.
     */
    public int getTotalSerialSize() {
        return getTotalSerialCount() * getSegmentRowCount();
    }

    public int getActiveSerialSize() {
        return getSerialActiveCount() * getSegmentRowCount();
    }

    /**
     * The total width of the CCD in pixels
     *
     * @return The total width in pixels, including active+overscan pixels
     */
    public int getTotalParallelSize() {
        return getTotalParallelCount() * getSegmentColumnCount();
    }

    public int getActiveParallelSize() {
        return getParallelActiveCount() * getSegmentColumnCount();
    }

    /**
     * Create a CCDTransform which considers all pixels for the absolute x,y
     * coordinates.
     *
     * @return The CCDTransform for global coordinates
     */
    public CCDTransform getGlobalTransform() {
        return new GlobalTransform();
    }

    /**
     * Create a CCDTransform which only considers active pixels for the absolute
     * x,y coordinates. When using this transform PixelType will always be
     * active. Setting a CCD coordinate which does not correspond to an active
     * pixel will result in an error. The {@link CCDTransform#getGlobalX()} and
     * {@link CCDTransform#getGlobalY()} methods can be used to obtain the
     * global coordinate corresponding to the x,y given.
     *
     * @return The CCDTransform for active coordinates.
     */
    public CCDTransform getActiveTransform() {
        return new ActiveTransform();
    }

    /**
     * Get the CCDSegment corresponding to the given row and column
     *
     * @param row The input row
     * @param column The input column
     * @return The CCDSegment and the specified row and column.
     */
    public CCDSegment getSegment(int row, int column) {
        assert (row >= 0 && row < segmentRowCount);
        assert (column >= 0 && column < segmentColumnCount);
        return channelMap[column * segmentRowCount + row];
    }

    /**
     * The list of segments in readout order.
     *
     * @return An unmodifiable list of segments, with item 0 corresponding to
     * amplifier 1.
     */
    public List<CCDSegment> getSegments() {
        return Collections.unmodifiableList(segments);
    }

    private static String pair(String a, String b) {
        return "[" + a + "," + b + "]";
    }

    private static String range(int a, int b) {
        return a + ":" + b;
    }

    private static String range(int a, int b, boolean flip) {
        return flip ? range(b, a) : range(a, b);
    }

    /**
     * Used when constructing the geometry to add segments (amplifiers) to the
     * geometry.
     *
     * @param channel The channel number (amplifier number) of the segment
     * (numbered from 1)
     * @param row The row containing the segment, numbered from 0 in the +Y
     * direction
     * @param column The column containing the segment, numbered from 0 in the
     * +X direction
     * @param readoutOrder The readout direction for the amplifier.
     */
    private void addSegment(int channel, int row, int column, ReadoutOrder readoutOrder) {
        final CCDSegment segment = new CCDSegment(channel, row, column, readoutOrder);
        segments.add(segment);
        channelMap[column * segmentRowCount + row] = segment;
    }

    /**
     * Create the BIASSEC fits header
     *
     * @return The value for BIASSEC
     */
    private String getBiasSec() {
        return pair(range(getSerialActiveCount() + getSerialPrescanCount() + 1,
                getSerialActiveCount() + getSerialPrescanCount() + getSerialOverscanCount()),
                range(1, getParallelActiveCount()));
    }

    /**
     * Create the DATASEC fits header
     *
     * @return The value for DATASEC
     */
    private String getDataSec(boolean treatOverAndUnderscanAsActive) {
        if (treatOverAndUnderscanAsActive) {
            return pair(range(1, getTotalSerialCount()), range(1, getTotalParallelCount()));
        } else {
            return pair(range(getSerialPrescanCount() + 1, getSerialActiveCount() + getSerialPrescanCount()),
                    range(1, getParallelActiveCount()));
        }
    }

    /**
     * Create the DETSIZE fits header
     *
     * @return The value for DETSIZE
     */
    private String getDetSize() {
        return pair(range(1, getActiveSerialSize()), range(1, getActiveParallelSize()));
    }

    /**
     * A CCDTransform which only considers active pixels for the absolute x,y
     * coordinates. When using this transform PixelType will always be active.
     * Setting a CCD coordinate which does not correspond to an active pixel
     * will result in an error. The {@link #getGlobalX()} and
     * {@link #getGlobalY()} methods can be used to obtain the global coordinate
     * corresponding to the x,y given.
     */
    private class ActiveTransform implements CCDTransform {

        private CCDSegment segment;
        private int serial;
        private int parallel;
        private int x;
        private int y;
        private int globalX;
        private int globalY;

        @Override
        public void setXY(int x, int y) {
            assert (x >= 0 && x < getActiveParallelSize());
            assert (y >= 0 && y < getActiveSerialSize());
            this.x = x;
            this.y = y;
            int column = x / parallelActive;
            int row = y / serialActive;
            segment = CCDGeometry.this.getSegment(row, column);
            globalY = y + segment.row * (serialPrescan + serialOverscan) + (segment.getReadout().isDown() ? serialPrescan : serialOverscan);
            globalX = x + segment.column * parallelOverscan + (!segment.getReadout().isRight() ? 0 : parallelOverscan);
            y %= serialActive;
            x %= parallelActive;
            serial = serialPrescan + (!segment.getReadout().isDown() ? y : serialActive - 1 - y);
            parallel = !segment.getReadout().isRight() ? x : parallelActive - 1 - x;
        }

        @Override
        public CCDSegment getSegment() {
            return segment;
        }

        @Override
        public int getSerial() {
            return serial;
        }

        @Override
        public int getParallel() {
            return parallel;
        }

        @Override
        public PixelType getPixelType() {
            return PixelType.ACTIVE;
        }

        @Override
        public int getGlobalX() {
            return globalX;
        }

        @Override
        public int getGlobalY() {
            return globalY;
        }

        @Override
        public String toString() {
            return "ActiveTransform{" + "segment=" + segment + ", serial=" + serial + ", parallel=" + parallel + ", x=" + x + ", y=" + y + '}';
        }

        @Override
        public void setSegmentSerialParallel(CCDSegment segment, int serial, int parallel) {
            assert (segment.getCCDGeometry() == CCDGeometry.this);
            assert (0 <= parallel && parallel < parallelActive);
            assert (serialPrescan <= serial && serial < serialPrescan + serialActive);
            this.segment = segment;
            this.serial = serial;
            this.parallel = parallel;
            serial -= serialPrescan;
            int yy = !segment.getReadout().isDown() ? serial : serialActive - 1 - serial;
            int xx = !segment.getReadout().isRight() ? parallel : parallelActive - 1 - parallel;
            yy += segment.getRow() * serialActive;
            xx += segment.getColumn() * parallelActive;
            this.x = xx;
            this.y = yy;
            globalY = y + segment.row * (serialPrescan + serialOverscan) + (segment.getReadout().isDown() ? serialPrescan : serialOverscan);
            globalX = x + segment.column * parallelOverscan + (!segment.getReadout().isRight() ? 0 : parallelOverscan);
        }

        @Override
        public int getX() {
            return x;
        }

        @Override
        public int getY() {
            return y;
        }
    }

    /**
     * A CCDTransform which considers all pixels for the absolute x,y
     * coordinates.
     */
    private class GlobalTransform implements CCDTransform {

        private CCDSegment segment;
        private int serial;
        private int parallel;
        private PixelType pixelType;
        private int x;
        private int y;

        @Override
        public void setXY(int x, int y) {
            assert (x >= 0 && x < getTotalParallelSize());
            assert (y >= 0 && y < getTotalSerialSize());
            this.x = x;
            this.y = y;
            int column = x / getTotalParallelCount();
            int row = y / getTotalSerialCount();
            segment = CCDGeometry.this.getSegment(row, column);
            y %= getTotalSerialCount();
            x %= getTotalParallelCount();
            serial = !segment.getReadout().isDown() ? y : getTotalSerialCount() - 1 - y;
            parallel = !segment.getReadout().isRight() ? x : getTotalParallelCount() - 1 - x;
            if (parallel > parallelActive) {
                pixelType = PixelType.PARALLEL_OVERSCAN;
            } else if (serial < serialPrescan) {
                pixelType = PixelType.SERIAL_PRESCAN;
            } else if (serial > serialPrescan + serialActive) {
                pixelType = PixelType.SERIAL_OVERSCAN;
            } else {
                pixelType = PixelType.ACTIVE;
            }
        }

        @Override
        public void setSegmentSerialParallel(CCDSegment segment, int serial, int parallel) {
            assert (segment.getCCDGeometry() == CCDGeometry.this);
            assert (0 <= parallel && parallel < getTotalParallelCount());
            assert (0 <= serial && serial < getTotalSerialCount());
            this.segment = segment;
            this.serial = serial;
            this.parallel = parallel;
            if (parallel > parallelActive) {
                pixelType = PixelType.PARALLEL_OVERSCAN;
            } else if (serial < serialPrescan) {
                pixelType = PixelType.SERIAL_PRESCAN;
            } else if (serial > serialPrescan + serialActive) {
                pixelType = PixelType.SERIAL_OVERSCAN;
            } else {
                pixelType = PixelType.ACTIVE;
            }
            int yy = !segment.getReadout().isDown() ? serial : getTotalSerialCount() - 1 - serial;
            int xx = !segment.getReadout().isRight() ? parallel : getTotalParallelCount() - 1 - parallel;
            yy += segment.getRow() * getTotalSerialCount();
            xx += segment.getColumn() * getTotalParallelCount();
            this.x = xx;
            this.y = yy;
        }

        @Override
        public CCDSegment getSegment() {
            return segment;
        }

        @Override
        public int getSerial() {
            return serial;
        }

        @Override
        public int getParallel() {
            return parallel;
        }

        @Override
        public PixelType getPixelType() {
            return pixelType;
        }

        @Override
        public String toString() {
            return "GlobalTransform{" + "segment=" + segment + ", serial=" + serial + ", parallel=" + parallel + ", pixelType=" + pixelType + ", x=" + x + ", y=" + y + '}';
        }

        @Override
        public int getGlobalX() {
            return x;
        }

        @Override
        public int getGlobalY() {
            return y;
        }

        @Override
        public int getX() {
            return x;
        }

        @Override
        public int getY() {
            return y;
        }

    }

    /**
     * Define the readout order for a particular amplifier. In this context
     * <ul>
     * <li>Left means +x</li>
     * <li>Right means -x</li>
     * <li>Up means -y</li>
     * <li>Down means +y</li>
     * </ul>
     * So DownRight means that the serial is readout Down (+y) and the parallel
     * is readout Right (-x).
     */
    public enum ReadoutOrder {

        DownRight(true, true), UpRight(false, true), DownLeft(true, false), UpLeft(false, false);

        private final boolean down;
        private final boolean right;

        private ReadoutOrder(boolean down, boolean right) {
            this.down = down;
            this.right = right;
        }

        public boolean isDown() {
            return down;
        }

        public boolean isRight() {
            return right;
        }

    };

    /**
     * Create fits headers for the primary header.
     * @return A map containing the fits header names and values.
     */
    public Map<String, Object> getPrimaryHeaders() {
        Map<String, Object> primaryMetaData = new HashMap<>();
        primaryMetaData.put("DETSIZE", getDetSize());
        return primaryMetaData;
    }

    /**
     * Class representing a single segment (amplifier) within a CCD.
     */
    public class CCDSegment {

        private ReadoutOrder readout;
        private int channel;
        private int row;
        private int column;

        private CCDSegment(int channel, int row, int column, ReadoutOrder readout) {
            this.readout = readout;
            this.channel = channel;
            this.row = row;
            this.column = column;
        }

        /**
         * Create fits headers for this segment.
         *
         * @return A map containing the fits header names and values.
         */
        public Map<String, Object> getSegmentHeaders() {
            return getSegmentHeaders(false);
        }

        /**
         * Create fits headers for this segment. Optionally create distorted
         * headers so that over and underscan regions can be viewed in DS9, as
         * requested by users at BNL.
         *
         * @param treatOverAndUnderscanAsActive If <code>true</code> then the
         * DATASEC and DETSEC headers are modified to treat all pixels as
         * active, and the BIASSEC keyword is omitted. Note that files written
         * in this way are not compliant with LCA-10140.
         * @return A map containing the fits header names and values.
         */
        public Map<String, Object> getSegmentHeaders(boolean treatOverAndUnderscanAsActive) {

            Map<String, Object> imageMetaData = new HashMap<>();

            imageMetaData.put("EXTNAME", String.format("Segment%01d%01d", column, row));
            imageMetaData.put("CHANNEL", channel);
            imageMetaData.put("DATASEC", getDataSec(treatOverAndUnderscanAsActive));
            imageMetaData.put("DETSEC", getDetSec(treatOverAndUnderscanAsActive));
            if (!treatOverAndUnderscanAsActive) {
                imageMetaData.put("BIASSEC", getBiasSec());
            }
            imageMetaData.put("CCDSUM", "1 1");
            imageMetaData.put("LTV1", getLTV1());
            imageMetaData.put("LTV2", getLTV2());
            imageMetaData.put("LTM1_1", getLTM11()); // delta along X axis
            imageMetaData.put("LTM2_2", getLTM22()); // delta along Y axis
            //FIXME: Waiting for input from Jim and Seth
//            imageMetaData.put("DTV1", 0); // detector transformation vector
//            imageMetaData.put("DTV2", 0); // detector transformation vector
//            imageMetaData.put("DTM1_1", 1); // detector transformation matrix
//            imageMetaData.put("DTM2_2", 1); // detector transformation matrix

            // WCS Keywords (from Stuart)
            imageMetaData.put("WCSNAME", "PRIMARY");
            imageMetaData.put("CTYPE1", "CCD_X");
            imageMetaData.put("CTYPE2", "CCD_Y");
            imageMetaData.put("CUNIT1", "pixel");
            imageMetaData.put("CUNIT2", "pixel");
            imageMetaData.put("CD1_1", getCD11());
            imageMetaData.put("CD1_2", 0.0);
            imageMetaData.put("CD2_1", 0.0);
            imageMetaData.put("CD2_2", getCD22());
            imageMetaData.put("CRPIX1", getCRPIX1());
            imageMetaData.put("CRPIX2", getCRPIX2());

            return imageMetaData;
        }

        /**
         * Generate the DETSEC fits header
         *
         * @return The value for DETSEC
         */
        private String getDetSec(boolean treatOverAndUnderscanAsActive) {
            if (treatOverAndUnderscanAsActive) {
                return pair(range(getTotalSerialCount() * row + 1,
                        getTotalSerialCount() * (row + 1), readout.isDown()),
                        range(getTotalParallelCount() * column + 1,
                                getTotalParallelCount() * (column + 1),
                                readout.isRight()));
            } else {
                return pair(range(getSerialActiveCount() * row + 1,
                        getSerialActiveCount() * (row + 1), readout.isDown()),
                        range(getParallelActiveCount() * column + 1,
                                getParallelActiveCount() * (column + 1),
                                readout.isRight()));
            }
        }

        private double getLTV1() {
            return readout.isDown() ? getSerialActiveCount() * (row + 1) : -(getSerialActiveCount() * row);
        }

        private double getLTV2() {
            return readout.isRight() ? getParallelActiveCount() * (column + 1) : getParallelActiveCount() * column;
        }

        private double getLTM11() {
            return readout.isDown() ? -1.0 : 1.0;
        }

        private double getLTM22() {
            return readout.isRight() ? -1.0 : 1.0;
        }

        private double getCD11() {
            return readout.isDown() ? -1.0 : 1.0;
        }

        private double getCD22() {
            return readout.isRight() ? -1.0 : 1.0;
        }

        private double getCRPIX1() {
            return readout.isDown() ? getTotalSerialCount() * (row + 1) + 1 : -(getTotalSerialCount() * row);
        }

        private double getCRPIX2() {
            return readout.isRight() ? getTotalParallelCount() * (column + 1) + 1: getTotalParallelCount() * column;
        }

        public int getChannel() {
            return channel;
        }

        public ReadoutOrder getReadout() {
            return readout;
        }

        public int getRow() {
            return row;
        }

        public int getColumn() {
            return column;
        }

        /**
         * Get the CCD geometry of which the segment is a part
         *
         * @return The CCDGeometry
         */
        public CCDGeometry getCCDGeometry() {
            return CCDGeometry.this;
        }

        @Override
        public String toString() {
            return "CCDSegment{" + "readout=" + readout + ", channel=" + channel + ", row=" + row + ", column=" + column + '}';
        }

    }
}
