package org.lsst.ccs.utilities.ccd;

import org.lsst.ccs.utilities.ccd.CCDGeometry.CCDSegment;

/**
 * Allows transforms between CCD coordinate system and CCD segment (amplifier) 
 * coordinate systems. The CCD coordinate system uses the conventions of the camera coordinate
 * system (confusingly called CCS).
 * Any set method may be used to set the currently selected coordinate, and all get 
 * methods will then return coordinates corresponding to the selected coordinate. The
 * precise meaning of the x,y coordinates depends on which method was used to obtain
 * the transform.
 * @see CCDGeometry#getGlobalTransform() 
 * @see CCDGeometry#getActiveTransform() 
 * @author tonyj
 */
public interface CCDTransform {

    /**
     * Enumerate the different types of pixel that can be read out.
     */
    public enum PixelType {
        /**
         * An active (signal) pixel 
         */
        ACTIVE, SERIAL_PRESCAN, SERIAL_OVERSCAN, PARALLEL_OVERSCAN}
    /**
     * Select the absolute x,y CCD coordinate.
     * @param x The x coordinate in the CCD coordinate system
     * @param y The u coordinate in the CCD coordinate system
     */
    void setXY(int x, int y);
    /**
     * Get the CCD x coordinate
     * @return The selected CCD x coordinate
     */
    int getX();
    /**
     * Get the CCD selected y coordinate system
     * @return The CCD y coordinate
     */
    int getY();
    /**
     * Set the current segment, and serial and parallel pixel position within
     * the segment (in segment readout order).
     * @param segment The CCD segment (amplifier)
     * @param serial The serial coordinate within the CCD segment
     * @param parallel The parallel coordinate within the CCD segment
     */
    void setSegmentSerialParallel(CCDSegment segment, int serial, int parallel);
    /**
     * The CCD segment corresponding to the currently selected coordinate.
     * @return The segment
     */
    CCDSegment getSegment();
    /**
     * The serial position of the currently selected pixel within the current CCD segment.
     * @return The serial coordinate
     */
    int getSerial();
    /**
     * The parallel coordinate of the currently selected pixel within the current CCD segment.
     * @return The parallel coordinate
     */
    int getParallel();
    /**
     * The type of pixel corresponding to the currently selected coordinate.
     * @return The pixel type
     */
    PixelType getPixelType();
    /**
     * The global coordinate corresponding to the currently selected coordinate. 
     * If the CCDTransform was obtained via {@link CCDGeometry#getGlobalTransform()} then 
     * it will be equivalent the the x coordinate, but if it was obtained via 
     * {@link CCDGeometry#getActiveParallelSize()} it will return the global x coordinate
     * corresponding to the currently selected active pixel.
     * @return The global x coordinate
     */
    int getGlobalX();
    /**
     * The global coordinate corresponding to the currently selected coordinate. 
     * If the CCDTransform was obtained via {@link CCDGeometry#getGlobalTransform()} then 
     * it will be equivalent the the y coordinate, but if it was obtained via 
     * {@link CCDGeometry#getActiveParallelSize()} it will return the global y coordinate
     * corresponding to the currently selected active pixel.
     * @return The global y coordinate
     */ 
    int getGlobalY();
}
