package org.lsst.ccs.utilities.image;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.util.Collection;
import java.util.Collections;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import nom.tam.fits.FitsException;

/**
 * A specification for a fits file header. Typically read from a .spec file.
 *
 * @author tonyj
 */
public class HeaderSpecification {

    private String name;
    private final Map<String, HeaderLine> headers = new LinkedHashMap<>();
    private final static Pattern expressionPattern = Pattern.compile("\\$\\{(?:(\\w+)\\.)?([\\w/]+)}");
    private final static Pattern linePattern = Pattern.compile("\\s*(\\S+)\\s+(\\w+)\\s+(?:\"(.*)\"|(\\S+))\\s*(.*)");

    public enum DataType {

        Boolean, Integer, String, Float, Date, MJD
    };

    public HeaderSpecification(String name, InputStream in) throws IOException {
        this(name, new InputStreamReader(in));
    }

    public HeaderSpecification(String name, Reader in) throws IOException {
        this.name = name;
        BufferedReader reader = in instanceof BufferedReader
                ? (BufferedReader) in : new BufferedReader(in);
        for (;;) {
            String line = reader.readLine();
            if (line == null) {
                break;
            } else if (line.startsWith("#")) {
                continue;
            }
            try {
                HeaderLine headerLine = new HeaderLine(line);
                headers.put(headerLine.getKeyword(), headerLine);
            } catch (IOException x) {
                throw new IOException("Error while reading line:\n\t" + line, x);
            }
        }

    }

    public String getName() {
        return name;
    }

    public HeaderLine getHeader(String keyWord) {
        return headers.get(keyWord);
    }
    
    public Collection<HeaderLine> getHeaders() {
        return Collections.unmodifiableCollection(headers.values());
    }

    public static class HeaderLine {

        private final String keyword;
        private final DataType dataType;
        private final String comment;
        private final String metaName;
        private final Object value;
        private final String metaMap;
        private final boolean isExpression;

        private HeaderLine(String line) throws IOException {
            try {
                Matcher lineMatcher = linePattern.matcher(line);
                if (!lineMatcher.matches()) {
                    throw new IOException("Invalid line in header specification");
                }
                keyword = lineMatcher.group(1);
                dataType = DataType.valueOf(lineMatcher.group(2));
                String valueExpression = lineMatcher.group(3) == null ? lineMatcher.group(4) : lineMatcher.group(3);
                Matcher matcher = expressionPattern.matcher(valueExpression);
                isExpression = matcher.matches();
                if (isExpression) {
                    metaMap = matcher.group(1);
                    metaName = matcher.group(2);
                    value = null;
                } else {
                    metaMap = null;
                    metaName = null;
                    value = coerce(valueExpression,dataType);
                }
                comment = lineMatcher.group(5);
            } catch (IllegalArgumentException | FitsException x) {
                throw new IOException("Illegal token found while reading header specification", x);
            }
        }

        public String getKeyword() {
            return keyword;
        }

        public DataType getDataType() {
            return dataType;
        }

        public String getComment() {
            return comment;
        }
        
        public String getMetaName() {
            return metaName;
        }

        Object getValue(Map<String, Map<String, Object>> metaData) {
            if (isExpression) {
                return findMetaDataValue(metaMap, metaName, metaData);
            } else {
                return value;
            }
        }

        private Object findMetaDataValue(String map, String name, Map<String, Map<String, Object>> metaDataSet) {
            return MetaDataSet.getValue(metaDataSet, map, name);
        }

        private Object coerce(String valueExpression, DataType dataType) throws FitsException, NumberFormatException {
            switch (dataType) {
                case Integer: return Integer.decode(valueExpression);
                case Float: return Double.valueOf(valueExpression);
                case Boolean: return Boolean.valueOf(valueExpression);
                case Date: return DateUtils.convertStringToDate(valueExpression);
                case MJD: return Double.valueOf(valueExpression);
                default: return valueExpression;
            }
        }
    }
}
