package org.lsst.ccs.utilities.ccd;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Defines the geometry of a CCD. See LCA-10103, in particular the diagram on
 * page 7.
 *
 * @see <a href="http://ls.st/lca-10103" target="_top">LCA-10103</a>
 */
public class CCDGeometry {

    private final int segmentRowCount;
    private final int segmentColumnCount;
    private final int serialPrescan;
    private final int parallelActive;
    private final int serialActive;
    private final int serialOverscan;
    private final int parallelOverscan;
    private final List<CCDSegment> segments;
    private final CCDSegment[] channelMap;

    /**
     * Create a CCDGeomtry object. To create geometry objects for standard CCD
     * types see the {@link CCDType} class.
     *
     * @param segmentRowCount The number of segments in the horizonal (serial)
     * direction
     * @param segmentColumnCount The number of segments in the vertical
     * (parallel) direction
     * @param serialActive The number of active pixels in the serial direction
     * @param parallelActive The number active pixels in the parallel direction
     * @param serialPrescan The number of serial prescan pixels for each
     * segment.
     * @param serialOverscan The number of overscan pixels in the serial
     * direction
     * @param parallelOverscan The number of overscan pixels in the parallel
     * direction
     */
    public CCDGeometry(int segmentRowCount, int segmentColumnCount, int serialActive, int parallelActive, int serialPrescan, int serialOverscan, int parallelOverscan) {
        this.segmentRowCount = segmentRowCount;
        this.segmentColumnCount = segmentColumnCount;
        this.serialPrescan = serialPrescan;
        this.parallelActive = parallelActive;
        this.serialActive = serialActive;
        this.serialOverscan = serialOverscan;
        this.parallelOverscan = parallelOverscan;
        this.segments = new ArrayList<>(segmentRowCount * segmentColumnCount);
        this.channelMap = new CCDSegment[segmentRowCount * segmentColumnCount];
    }

    public int getSerialPrescanCount() {
        return serialPrescan;
    }

    public int getParallelActiveCount() {
        return parallelActive;
    }

    public int getSerialActiveCount() {
        return serialActive;
    }

    public int getSerialOverscanCount() {
        return serialOverscan;
    }

    public int getParallelOverscanCount() {
        return parallelOverscan;
    }

    public int getSegmentRowCount() {
        return segmentRowCount;
    }

    public int getSegmentColumnCount() {
        return segmentColumnCount;
    }

    /**
     * Total serial pixels per segment.
     *
     * @return The total number of active+prescan+overscan pixels in the serial
     * direction.
     */
    public int getTotalSerialCount() {
        return getSerialPrescanCount() + getSerialActiveCount() + getSerialOverscanCount();
    }

    /**
     * Total parallel pixels per segment.
     *
     * @return The total number of active+overscan pixels in the parallel
     * direction.
     */
    public int getTotalParallelCount() {
        return getParallelActiveCount() + getParallelOverscanCount();
    }

    /**
     * The total height of the CCD in pixels
     *
     * @return The total height in pixels, including active+prescan+overscan
     * pixels.
     */
    public int getTotalSerialSize() {
        return getTotalSerialCount() * getSegmentRowCount();
    }

    public int getActiveSerialSize() {
        return getSerialActiveCount() * getSegmentRowCount();
    }

    /**
     * The total width of the CCD in pixels
     *
     * @return The total width in pixels, including active+overscan pixels
     */
    public int getTotalParallelSize() {
        return getTotalParallelCount() * getSegmentColumnCount();
    }

    public int getActiveParallelSize() {
        return getParallelActiveCount() * getSegmentColumnCount();
    }

    /**
     * Create a CCDTransform which considers all pixels for the absolute x,y
     * coordinates.
     *
     * @return The CCDTransform for global coordinates
     */
    public CCDTransform getGlobalTransform() {
        return new GlobalTransform();
    }

    /**
     * Create a CCDTransform which only considers active pixels for the absolute
     * x,y coordinates. When using this transform PixelType will always be
     * active. Setting a CCD coordinate which does not correspond to an active
     * pixel will result in an error. The {@link CCDTransform#getGlobalX()} and
     * {@link CCDTransform#getGlobalY()} methods can be used to obtain the
     * global coordinate corresponding to the x,y given.
     *
     * @return The CCDTransform for active coordinates.
     */
    public CCDTransform getActiveTransform() {
        return new ActiveTransform();
    }

    /**
     * Get the CCDSegment corresponding to the given row and column
     *
     * @param row The input row
     * @param column The input column
     * @return The CCDSegment and the specified row and column.
     */
    public CCDSegment getSegment(int row, int column) {
        assert (row >= 0 && row < segmentRowCount);
        assert (column >= 0 && column < segmentColumnCount);
        return channelMap[column * segmentRowCount + row];
    }

    /**
     * The list of segments in readout order.
     *
     * @return An unmodifiable list of segments, with item 0 corresponding to
     * amplifier 1.
     */
    public List<CCDSegment> getSegments() {
        return Collections.unmodifiableList(segments);
    }

    private static String pair(String a, String b) {
        return "[" + a + "," + b + "]";
    }

    private static String range(int a, int b) {
        return a + ":" + b;
    }

    private static String range(int a, int b, boolean flip) {
        return flip ? range(b, a) : range(a, b);
    }

    /**
     * Used when constructing the geometry to add segments (amplifiers) to the
     * geometry.
     *
     * @param channel The channel number (amplifier number) of the segment
     * (numbered from 1)
     * @param row The row containing the segment, numbered from 0 in the +Y
     * direction
     * @param column The column containing the segment, numbered from 0 in the
     * +X direction
     * @param readoutOrder The readout direction for the amplifier.
     */
    void addSegment(int channel, int row, int column, ReadoutOrder readoutOrder) {
        final CCDSegment segment = new CCDSegment(channel, row, column, readoutOrder);
        segments.add(segment);
        channelMap[column * segmentRowCount + row] = segment;
    }

    /**
     * Create the BIASSEC fits header
     *
     * @return The value for BIASSEC
     */
    private String getBiasSec() {
        return pair(range(getSerialActiveCount() + getSerialPrescanCount() + 1,
                getSerialActiveCount() + getSerialPrescanCount() + getSerialOverscanCount()),
                range(1, getParallelActiveCount()));
    }

    /**
     * Create the DATASEC fits header
     *
     * @return The value for DATASEC
     */
    private String getDataSec(boolean treatOverAndUnderscanAsActive) {
        if (treatOverAndUnderscanAsActive) {
            return pair(range(1, getTotalSerialCount()), range(1, getTotalParallelCount()));
        } else {
            return pair(range(getSerialPrescanCount() + 1, getSerialActiveCount() + getSerialPrescanCount()),
                    range(1, getParallelActiveCount()));
        }
    }

    /**
     * Create the DETSIZE fits header
     *
     * @return The value for DETSIZE
     */
    private String getDetSize() {
        return pair(range(1, getTotalSerialSize()), range(1, getTotalParallelSize()));
    }

    /**
     * A CCDTransform which only considers active pixels for the absolute x,y
     * coordinates. When using this transform PixelType will always be active.
     * Setting a CCD coordinate which does not correspond to an active pixel
     * will result in an error. The {@link #getGlobalX()} and
     * {@link #getGlobalY()} methods can be used to obtain the global coordinate
     * corresponding to the x,y given.
     */
    private class ActiveTransform implements CCDTransform {

        private CCDSegment segment;
        private int serial;
        private int parallel;
        private int x;
        private int y;
        private int globalX;
        private int globalY;

        @Override
        public void setXY(int x, int y) {
            assert (x >= 0 && x < getActiveParallelSize());
            assert (y >= 0 && y < getActiveSerialSize());
            this.x = x;
            this.y = y;
            int column = x / parallelActive;
            int row = y / serialActive;
            segment = CCDGeometry.this.getSegment(row, column);
            globalY = y + segment.row * (serialPrescan + serialOverscan) + (segment.getReadout().isDown() ? serialPrescan : serialOverscan);
            globalX = x + segment.column * parallelOverscan + (!segment.getReadout().isRight() ? 0 : parallelOverscan);
            y %= serialActive;
            x %= parallelActive;
            serial = serialPrescan + (!segment.getReadout().isDown() ? y : serialActive - 1 - y);
            parallel = !segment.getReadout().isRight() ? x : parallelActive - 1 - x;
        }

        @Override
        public CCDSegment getSegment() {
            return segment;
        }

        @Override
        public int getSerial() {
            return serial;
        }

        @Override
        public int getParallel() {
            return parallel;
        }

        @Override
        public PixelType getPixelType() {
            return PixelType.ACTIVE;
        }

        @Override
        public int getGlobalX() {
            return globalX;
        }

        @Override
        public int getGlobalY() {
            return globalY;
        }

        @Override
        public String toString() {
            return "ActiveTransform{" + "segment=" + segment + ", serial=" + serial + ", parallel=" + parallel + ", x=" + x + ", y=" + y + '}';
        }

        @Override
        public void setSegmentSerialParallel(CCDSegment segment, int serial, int parallel) {
            assert (segment.getCCDGeometry() == CCDGeometry.this);
            assert (0 <= parallel && parallel < parallelActive);
            assert (serialPrescan <= serial && serial < serialPrescan + serialActive);
            this.segment = segment;
            this.serial = serial;
            this.parallel = parallel;
            serial -= serialPrescan;
            int yy = !segment.getReadout().isDown() ? serial : serialActive - 1 - serial;
            int xx = !segment.getReadout().isRight() ? parallel : parallelActive - 1 - parallel;
            yy += segment.getRow() * serialActive;
            xx += segment.getColumn() * parallelActive;
            this.x = xx;
            this.y = yy;
            globalY = y + segment.row * (serialPrescan + serialOverscan) + (segment.getReadout().isDown() ? serialPrescan : serialOverscan);
            globalX = x + segment.column * parallelOverscan + (!segment.getReadout().isRight() ? 0 : parallelOverscan);
        }

        @Override
        public int getX() {
            return x;
        }

        @Override
        public int getY() {
            return y;
        }
    }

    /**
     * A CCDTransform which considers all pixels for the absolute x,y
     * coordinates.
     */
    private class GlobalTransform implements CCDTransform {

        private CCDSegment segment;
        private int serial;
        private int parallel;
        private PixelType pixelType;
        private int x;
        private int y;

        @Override
        public void setXY(int x, int y) {
            assert (x >= 0 && x < getTotalParallelSize());
            assert (y >= 0 && y < getTotalSerialSize());
            this.x = x;
            this.y = y;
            int column = x / getTotalParallelCount();
            int row = y / getTotalSerialCount();
            segment = CCDGeometry.this.getSegment(row, column);
            y %= getTotalSerialCount();
            x %= getTotalParallelCount();
            serial = !segment.getReadout().isDown() ? y : getTotalSerialCount() - 1 - y;
            parallel = !segment.getReadout().isRight() ? x : getTotalParallelCount() - 1 - x;
            if (parallel > parallelActive) {
                pixelType = PixelType.PARALLEL_OVERSCAN;
            } else if (serial < serialPrescan) {
                pixelType = PixelType.SERIAL_PRESCAN;
            } else if (serial > serialPrescan + serialActive) {
                pixelType = PixelType.SERIAL_OVERSCAN;
            } else {
                pixelType = PixelType.ACTIVE;
            }
        }

        @Override
        public void setSegmentSerialParallel(CCDSegment segment, int serial, int parallel) {
            assert (segment.getCCDGeometry() == CCDGeometry.this);
            assert (0 <= parallel && parallel < getTotalParallelCount());
            assert (0 <= serial && serial < getTotalSerialCount());
            this.segment = segment;
            this.serial = serial;
            this.parallel = parallel;
            if (parallel > parallelActive) {
                pixelType = PixelType.PARALLEL_OVERSCAN;
            } else if (serial < serialPrescan) {
                pixelType = PixelType.SERIAL_PRESCAN;
            } else if (serial > serialPrescan + serialActive) {
                pixelType = PixelType.SERIAL_OVERSCAN;
            } else {
                pixelType = PixelType.ACTIVE;
            }
            int yy = !segment.getReadout().isDown() ? serial : getTotalSerialCount() - 1 - serial;
            int xx = !segment.getReadout().isRight() ? parallel : getTotalParallelCount() - 1 - parallel;
            yy += segment.getRow() * getTotalSerialCount();
            xx += segment.getColumn() * getTotalParallelCount();
            this.x = xx;
            this.y = yy;
        }

        @Override
        public CCDSegment getSegment() {
            return segment;
        }

        @Override
        public int getSerial() {
            return serial;
        }

        @Override
        public int getParallel() {
            return parallel;
        }

        @Override
        public PixelType getPixelType() {
            return pixelType;
        }

        @Override
        public String toString() {
            return "GlobalTransform{" + "segment=" + segment + ", serial=" + serial + ", parallel=" + parallel + ", pixelType=" + pixelType + ", x=" + x + ", y=" + y + '}';
        }

        @Override
        public int getGlobalX() {
            return x;
        }

        @Override
        public int getGlobalY() {
            return y;
        }

        @Override
        public int getX() {
            return x;
        }

        @Override
        public int getY() {
            return y;
        }

    }

    /**
     * Define the readout order for a particular amplifier. In this context
     * <ul>
     * <li>Left means +x</li>
     * <li>Right means -x</li>
     * <li>Up means -y</li>
     * <li>Down means +y</li>
     * </ul>
     * So DownRight means that the serial is readout Down (+y) and the parallel
     * is readout Right (-x).
     */
    public enum ReadoutOrder {

        DownRight(true, true), UpRight(false, true), DownLeft(true, false), UpLeft(false, false);

        private boolean down;
        private boolean right;

        private ReadoutOrder(boolean down, boolean right) {
            this.down = down;
            this.right = right;
        }

        public boolean isDown() {
            return down;
        }

        public boolean isRight() {
            return right;
        }

    };

    /**
     * Create fits headers for the primary header. 
     * @return A map containing the fits header names and values.
     */
    public Map<String, Object> getPrimaryHeaders() {
        Map<String, Object> primaryMetaData = new HashMap<>();
        primaryMetaData.put("DETSIZE", getDetSize());
        return primaryMetaData;
    }

    /**
     * Class representing a single segment (amplifier) within a CCD.
     */
    public class CCDSegment {

        private ReadoutOrder readout;
        private int channel;
        private int row;
        private int column;

        private CCDSegment(int channel, int row, int column, ReadoutOrder readout) {
            this.readout = readout;
            this.channel = channel;
            this.row = row;
            this.column = column;
        }

        /**
         * Create fits headers for this segment.
         *
         * @return A map containing the fits header names and values.
         */
        public Map<String, Object> getSegmentHeaders() {
            return getSegmentHeaders(false);
        }

        /**
         * Create fits headers for this segment. Optionally create distorted
         * headers so that over and underscan regions can be viewed in DS9, as
         * requested by users at BNL.
         *
         * @param treatOverAndUnderscanAsActive If <code>true</code> then the
         * DATASEC and DETSEC headers are modified to treat all pixels as
         * active, and the BIASSEC keyword is omitted. Note that files written
         * in this way are not compliant with LCA-10140.
         * @return A map containing the fits header names and values.
         */
        public Map<String, Object> getSegmentHeaders(boolean treatOverAndUnderscanAsActive) {

            Map<String, Object> imageMetaData = new HashMap<>();

            imageMetaData.put("EXTNAME", String.format("Segment%01d%01d", column, row));
            imageMetaData.put("CHANNEL", channel);
            imageMetaData.put("DATASEC", getDataSec(treatOverAndUnderscanAsActive));
            imageMetaData.put("DETSEC", getDetSec(treatOverAndUnderscanAsActive));
            if (!treatOverAndUnderscanAsActive) {
                imageMetaData.put("BIASSEC", getBiasSec());
            }
            imageMetaData.put("CCDSUM", "1 1");
            //FIXME: Waiting for input from Jim and Seth
//            imageMetaData.put("LTV1", getLTV1());
//            imageMetaData.put("LTV2", getLTV2());
//            imageMetaData.put("LTM1_1", getLTM1()); // delta along X axis
//            imageMetaData.put("LTM2_2", getLTM2()); // delta along Y axis
//            imageMetaData.put("DTV1", 0); // detector transformation vector
//            imageMetaData.put("DTV2", 0); // detector transformation vector
//            imageMetaData.put("DTM1_1", 1); // detector transformation matrix
//            imageMetaData.put("DTM2_2", 1); // detector transformation matrix

            return imageMetaData;
        }

        /**
         * Generate the DETSEC fits header
         *
         * @return The value for DETSEC
         */
        private String getDetSec(boolean treatOverAndUnderscanAsActive) {
            if (treatOverAndUnderscanAsActive) {
                return pair(range(getTotalSerialCount() * row + 1,
                        getTotalSerialCount() * (row + 1), readout.isDown()),
                        range(getTotalParallelCount() * (1 - column) + 1,
                                getTotalParallelCount() * (1 - column + 1),
                                !readout.isRight()));
            } else {
                return pair(range(getSerialActiveCount() * row + 1,
                        getSerialActiveCount() * (row + 1), readout.isDown()),
                        range(getParallelActiveCount() * (1 - column) + 1,
                                getParallelActiveCount() * (1 - column + 1),
                                !readout.isRight()));
            }
        }

        private double getLTV1() {
            return getTotalSerialCount();
        }

        private double getLTV2() {
            return 2.0 * getTotalParallelCount();
        }

        private double getLTM1() {
            return readout.isRight() ? -1.0 : 1.0;
        }

        private double getLTM2() {
            return readout.isDown() ? -1.0 : 1.0;
        }

        public int getChannel() {
            return channel;
        }

        public ReadoutOrder getReadout() {
            return readout;
        }

        public int getRow() {
            return row;
        }

        public int getColumn() {
            return column;
        }

        /**
         * Get the CCD geometry of which the segment is a part
         *
         * @return The CCDGeometry
         */
        public CCDGeometry getCCDGeometry() {
            return CCDGeometry.this;
        }

        @Override
        public String toString() {
            return "CCDSegment{" + "readout=" + readout + ", channel=" + channel + ", row=" + row + ", column=" + column + '}';
        }

    }
}
