package org.lsst.ccs.utilities.image;

import java.io.Closeable;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.Map;
import nom.tam.fits.BasicHDU;
import nom.tam.fits.FitsException;
import nom.tam.fits.FitsFactory;
import nom.tam.fits.FitsUtil;
import nom.tam.fits.Header;
import nom.tam.fits.HeaderCardException;
import nom.tam.util.BufferedFile;
import static org.lsst.ccs.utilities.image.HeaderSpecification.DataType.Float;
import org.lsst.ccs.utilities.image.HeaderSpecification.HeaderLine;
import org.lsst.ccs.utilities.image.ImageSet.Image;

/**
 * A utility for writing FITS files following LSST conventions.
 *
 * @author tonyj
 */
public class FitsFileWriter implements Closeable {

    private final BufferedFile bf;
    // The current write position for each extended HDU
    private final long[] position0, position;

    public enum BitsPerPixel {

        BIT16, BIT32
    };

    /**
     * Open an LSST FITS file for writing a CCD ImageSet.
     *
     * @param fileName The file to write to
     * @param images The ImageSet to write to the file. Note that this specifies
     * the images to write, not the actual data for the images.
     * @param metaData The meta-data maps to use to extract header info from
     * @param config The configuration which controls how meta data is written
     * to the file
     * @param bits The number of bits per pixel for images
     * @throws java.io.IOException
     * @throws nom.tam.fits.FitsException
     */
    public FitsFileWriter(String fileName, ImageSet images, Map<String, Map<String, Object>> metaData, Map<String, HeaderSpecification> config, BitsPerPixel bits) throws IOException, FitsException {

        position0 = new long[images.getImages().size() + 1];
        position = new long[images.getImages().size()];
        int[][] intDummyData = new int[1][1];
        short[][] shortDummyData = new short[1][1];
        Object[] tableDummyData = new Object[0];
        Object dummyData = bits == BitsPerPixel.BIT16 ? shortDummyData : intDummyData;
        bf = new BufferedFile(fileName, "rw");
        // Create primary header
        BasicHDU primary = BasicHDU.getDummyHDU();
        addMetaDataToHeader(primary, "primary", metaData, config);
        primary.getHeader().write(bf);
        
        // If necessary, create any additional extended HDU's here.
        //Create any extra BinTables from the specification
        FitsFactory.setUseAsciiTables(false);
        for (String key : config.keySet()) {
            if (!"primary".equals(key) && !"extended".equals(key)) {
                BasicHDU binary = FitsFactory.HDUFactory(tableDummyData);
                addMetaDataToHeader(binary, key, metaData, config);
                Header header = binary.getHeader();
                header.setXtension("BINTABLE");
                header.write(bf);
            }
        }

        //Create extension image headers for each image, and reserve space for data
        int i = 0;
        for (Image image : images.getImages()) {
            BasicHDU hdu = FitsFactory.HDUFactory(dummyData);
            addMetaDataToHeader(hdu, "extended", image.getMetaData(), config);
            Header header = hdu.getHeader();
            header.setXtension("IMAGE");
            header.setNaxis(1, image.getWidth());
            header.setNaxis(2, image.getHeight());
            position0[i] = bf.getFilePointer();
            header.write(bf);
            position[i] = bf.getFilePointer();
            long imageSize = (bits == BitsPerPixel.BIT16 ? 2l : 4l) * image.getWidth() * image.getHeight();
            bf.seek(bf.getFilePointer() + imageSize);
            FitsUtil.pad(bf, imageSize);
            i++;
        }
        position0[i] = bf.getFilePointer();
    }

    /**
     * Write the actual image data to the file. It is not necessary that all of
     * the data for the image be available at once, this method will write
     * whatever data is currently available in the byte buffer to the specified
     * image, and will keep track of how much has been written to each image to
     * allow more data to be written later. This method assumes the data is
     * given in the order it is to be written to the file. If any data
     * reordering is needed it needs to be done before calling this method.
     *
     * @param imageIndex The image to which this data is to be written
     * @param src The image
     * @throws IOException If an IOException is generated, or if more data is
     * sent than was expected for a particular image.
     */
    public void write(int imageIndex, ByteBuffer src) throws IOException {
        int length = src.remaining();
        if (length + position[imageIndex] > position0[imageIndex + 1]) {
            throw new IOException("Too much data written for image: " + imageIndex);
        }
        bf.seek(position[imageIndex]);
        if (src.hasArray()) {
            bf.write(src.array(), src.arrayOffset() + src.position(), src.remaining());
            src.position(src.limit());
        } else {
            while (src.remaining() > 0) {
                bf.write(src.get());
            }
        }
        position[imageIndex] += length;
    }

    @Override
    public void close() throws IOException {
        bf.close();
    }

    private void addMetaDataToHeader(BasicHDU hdu, String specName, Map<String, Map<String, Object>> metaData, Map<String, HeaderSpecification> config) throws HeaderCardException, IOException {
        HeaderSpecification spec = config.get(specName);
        if (spec == null) {
            throw new IOException("Missing specification for header: " + specName);
        }
        for (HeaderLine header : spec.getHeaders()) {
            Object value = header.getValue(metaData);
            try {
                if (value != null) {
                    switch (header.getDataType()) {
                        case Integer:
                            hdu.addValue(header.getKeyword(), ((Number) value).intValue(), header.getComment());
                            break;
                        case Float:
                            hdu.addValue(header.getKeyword(), ((Number) value).doubleValue(), header.getComment());
                            break;
                        case Boolean:
                            hdu.addValue(header.getKeyword(), ((Boolean) value).booleanValue(), header.getComment());
                            break;
                        default:
                            hdu.addValue(header.getKeyword(), String.valueOf(value), header.getComment());
                    }
                }
            } catch (ClassCastException x) {
                throw new IOException(String.format("Meta-data header %s with value %s(%s) cannot be converted to type %s", header.getKeyword(), value, value.getClass(), header.getDataType()));
            }
        }
    }
}
