package org.lsst.ccs.imagenaming.rest;

import java.io.IOException;
import java.sql.SQLException;
import java.time.Duration;
import java.util.List;
import java.util.TimeZone;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import javax.servlet.ServletContext;
import javax.ws.rs.GET;
import javax.ws.rs.Path;
import javax.ws.rs.Produces;
import javax.ws.rs.QueryParam;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import org.lsst.ccs.imagenaming.Controller;
import org.lsst.ccs.imagenaming.ImageName;
import org.lsst.ccs.imagenaming.ImageNameAssigner;
import org.lsst.ccs.imagenaming.Source;

/**
 *
 * @author tonyj
 */
@Path("/")
@Produces(MediaType.APPLICATION_JSON)
public class DataSource {

    private ImageNameAssigner imageNameAssigner;
    private static final Logger LOG = Logger.getLogger(DataSource.class.getName());

    @Context
    public void setServletContext(ServletContext context) throws IOException {
        String dbURL = "jdbc:h2:mem:test;MODE=MYSQL";
        if (context != null) {
            String url = context.getInitParameter("org.lsst.ccs.imagenaming.rest.dbURL");
            if (url != null) {
                dbURL = url;
            }
        }
        LOG.log(Level.INFO, "Using dbURL: {0}", dbURL);
        imageNameAssigner = new ImageNameAssigner(dbURL);
    }

    @GET
    @Path("imageNameService")
    public List<String> imageNameService(
            @QueryParam(value = "n") int n,
            @QueryParam(value = "source") Source source,
            @QueryParam(value = "sourceIndex") int sourceIndex) throws SQLException {
        if (n == 0) {
            n = 1;
        }
        if (source == null) {
            source = Source.GenericCamera;
        }
        final TimeZone timeZone = TimeZone.getTimeZone("UTC");
        try {
            final List<String> imageName = imageNameAssigner.assignImageNames(source, sourceIndex, Controller.OCS, timeZone, Duration.ofHours(12), n, null).stream().map(ImageName::toString).collect(Collectors.toList());
            LOG.log(Level.INFO, "Request n={0} source={1} sourceIndex={2} returning {3}", new Object[]{n, source, sourceIndex, imageName});
            return imageName;
        } finally {
            // Temporary, inefficient, fix for preventing database connection leaks
            // pending better understanding of servlet lifecycle
            imageNameAssigner.close();
            LOG.log(Level.INFO, "Using dbURL closed");
        }
    }
}
